<?php

/**
 * Retrieve report data from Google.
 */
class Advanced_Ads_AdSense_Report_Api {

	/**
	 * Report endpoint.
	 *
	 * @var string
	 */
	const ENDPOINT_URL = 'https://adsense.googleapis.com/v2/accounts/%pubid%/reports:generate?startDate.year=%SY%&startDate.month=%SM%&startDate.day=%SD%'
						 . '&endDate.year=%EY%&endDate.day=%ED%&endDate.month=%EM%&dimensions=%DIM%&dimensions=DATE&metrics=ESTIMATED_EARNINGS&reportingTimeZone=ACCOUNT_TIME_ZONE';

	/**
	 * Version of the AdSense Management API in use (for getting fresh data).
	 *
	 * @var string
	 */
	const API_VERSION = '2.0';

	/**
	 * Report type
	 *
	 * @var string
	 */
	private $type;

	/**
	 * The API access token or an error array.
	 *
	 * @var array|string
	 */
	private $access_token;

	/**
	 * The current connected AdSense account.
	 *
	 * @var string
	 */
	private $publisher_id;

	/**
	 * Instance constructor.
	 *
	 * @param string $type report type.
	 */
	public function __construct( $type ) {
		$publisher_id       = Advanced_Ads_AdSense_Data::get_instance()->get_adsense_id();
		$this->type         = $type;
		$this->access_token = Advanced_Ads_AdSense_MAPI::get_access_token( $publisher_id );
		$this->publisher_id = $publisher_id;
	}

	/**
	 * Checks if the current setup has an access token.
	 *
	 * @return bool true if there is a token.
	 */
	public function has_token() {
		return is_string( $this->access_token );
	}

	/**
	 * Get access token related error message.
	 *
	 * @return array Array of error messages.
	 */
	public function get_token_error() {
		return is_string( $this->access_token ) ? array() : $this->access_token;
	}

	/**
	 * Check if there is an error related to access tokens.
	 *
	 * @return bool true if any error happened when requesting an access token.
	 */
	public function has_token_error() {
		return ! is_string( $this->access_token );
	}

	/**
	 * Perform the actual call to Google for fresh data.
	 *
	 * @return array associative array with the response or with error data in case of failure.
	 */
	public function call_google() {
		$dimension  = $this->type === 'unit' ? 'AD_UNIT_ID' : 'DOMAIN_NAME';
		$today      = date_create( 'now' );
		$start_date = date_create( 'now' )->sub( date_interval_create_from_date_string( '28 days' ) );

		// Replace placeholder in the endpoint with actual arguments.
		$url = str_replace(
			array(
				'%pubid%',
				'%DIM%',
				'%SY%',
				'%SM%',
				'%SD%',
				'%EY%',
				'%EM%',
				'%ED%',
			),
			array(
				$this->publisher_id,
				$dimension,
				$start_date->format( 'Y' ),
				$start_date->format( 'n' ),
				$start_date->format( 'j' ),
				$today->format( 'Y' ),
				$today->format( 'n' ),
				$today->format( 'j' ),
			),
			self::ENDPOINT_URL
		);

		$headers = array(
			'Authorization' => 'Bearer ' . $this->access_token,
		);

		$response = wp_remote_get( $url, array( 'headers' => $headers ) );
		Advanced_Ads_AdSense_MAPI::log( 'Fetched AdSense Report from ' . $url );

		if ( is_wp_error( $response ) ) {
			return array(
				'status' => false,
				// translators: AdSense ID.
				'msg'    => sprintf( esc_html__( 'Error while retrieving report for "%s".', 'advanced-ads' ), $this->publisher_id ),
				'raw'    => $response->get_error_message(),
			);
		}

		$response_body = json_decode( $response['body'], true );

		if ( $response_body === null ) {
			return array(
				'status' => false,
				// translators: AdSense ID.
				'msg'    => sprintf( esc_html__( 'Invalid response while retrieving report for "%s".', 'advanced-ads' ), $this->publisher_id ),
				'raw'    => $response['body'],
			);
		}

		if ( ! isset( $response_body['totalMatchedRows'] ) ) {
			return array(
				'status' => false,
				// translators: AdSense ID.
				'msg'    => sprintf( esc_html__( 'Invalid response while retrieving report for "%s".', 'advanced-ads' ), $this->publisher_id ),
				'raw'    => $response['body'],
			);
		}

		$response_body['api_version'] = self::API_VERSION;
		$response_body['timestamp']  = time();

		return array(
			'status'   => true,
			'response_body' => $response_body,
		);
	}

}
