<?php
/**
 * Que Items List
 *
 * @package    BuddyPress Moderation Tools
 * @subpackage Admin
 * @copyright  Copyright (c) 2018, Brajesh Singh
 * @license    https://www.gnu.org/licenses/gpl.html GNU Public License
 * @author     Brajesh Singh
 * @since      1.0.0
 */

namespace PressThemes\ModerationTools\Admin;

use PressThemes\ModerationTools\Models\Queue;

// Do not allow direct access over web.
defined( 'ABSPATH' ) || exit;


/**
 * Class Queue_Items_Table
 */
class Queue_Items_Table extends \WP_List_Table {

	/**
	 * Flag vars.
	 *
	 * @var array
	 */
	private $args = array();

	/**
	 * Queue_Items_Table constructor.
	 *
	 * @param array $args Array of values.
	 */
	public function __construct( $args = array() ) {
		$this->args = $args;

		$parent_args = array(
			'singular' => 'queue_item',
			'plural'   => 'queue_items',
			'ajax'     => true,
		);

		parent::__construct( $parent_args );
	}

	/**
	 * Check user permissions
	 *
	 * @return bool
	 */
	public function ajax_user_can() {
		return current_user_can( 'manage_options' );
	}

	/**
	 * All logic goes here
	 */
	public function prepare_items() {
		$current_page = $this->get_pagenum();
		$per_page     = 20;

		$args = array(
			'per_page' => $per_page,
			'is_new'   => 1,
			'page'     => $current_page,
		);

		if ( ! empty( $this->args['type'] ) ) {
			$args['item_type'] = $this->args['type'];
		}

		if ( ! empty( $this->args['hidden'] ) ) {
			$args['is_hidden'] = $this->args['hidden'];
		}

		$args['orderby'] = 'updated_at';
		$args['order']   = 'DESC';

		$this->items = Queue::get( $args );

		unset( $args['per_page'] );
		unset( $args['offset'] );

		$this->set_pagination_args( array(
			'total_items' => Queue::count( $args ),
			'per_page'    => $per_page,
		) );
	}

	/**
	 * Render message when no items found
	 */
	public function no_items() {
		_e( 'No items in moderation.', 'buddypress-moderation-tools' );
	}

	/**
	 * Get views
	 *
	 * @return array
	 */
	protected function get_views() {
		$class = empty( $this->args['type'] ) ? 'current' : '';

		$views = array( 'all' => sprintf( '<a href="%s" class="%s">%s</a>', bpmts_admin_get_queue_url(), $class, __( 'All', 'buddypress-moderation-tools' ) ) );

		$reportable_items = bpmts_moderation_tools()->reportable_types->get_all();
		$is_hidden        = empty( $this->args['hidden'] ) ? 0 : 1;

		foreach ( $reportable_items as $item ) {
			if ( ! $item->active ) {
				continue;
			}

			$class = $item->type === $this->args['type'] ? 'current' : '';

			$views[ 'bpmts-type-' . $item->type ] = sprintf( '<a href="%s" class="%s">%s</a>', bpmts_admin_get_queue_url( array(
				'bpmts_type'  => $item->type,
				'bpmts_state' => $is_hidden,
			) ), $class, $item->label );
		}

		return $views;
	}

	/**
	 * Return bulk actions
	 *
	 * @return array
	 */
	protected function get_bulk_actions() {
		return array();
	}

	/**
	 * Get column info
	 *
	 * @return array
	 */
	public function get_columns() {
		return array(
			'cb'              => '<input type="checkbox" />',
			'thumbnail'       => __( 'Thumbnail', 'buddypress-moderation-tools' ),
			'item'            => __( 'Reported Item', 'buddypress-moderation-tools' ),
			'type'            => __( 'Type', 'buddypress-moderation-tools' ),
			'user_id'         => __( 'Related user', 'buddypress-moderation-tools' ),
			'current_reports' => __( 'New Reports', 'buddypress-moderation-tools' ),
			'total_reports'   => __( 'All Reports', 'buddypress-moderation-tools' ),
			'updated_at'      => __( 'Last reported', 'buddypress-moderation-tools' ),
		);
	}

	/**
	 * Sortable columns.
	 *
	 * @return array
	 */
	public function get_sortable_columns() {

		$sortable_columns = array(
			'user_id'    => array( 'user_id', false ),
			'type'       => array( 'type', false ),
			'updated_at' => array( 'updated_at', false ),
		);

		return $sortable_columns;
	}

	/**
	 * Mask action
	 *
	 * @param mixed $item Current queue id.
	 *
	 * @return string
	 */
	public function column_cb( $item ) {
		return sprintf(
			'<input type="checkbox" name="items[]" value="%d" />', $item->id
		);
	}

	/**
	 * Column data.
	 *
	 * @param Queue  $item que item.
	 * @param string $col column name.
	 *
	 * @return string
	 */
	public function column_default( $item, $col ) {

		switch ( $col ) {

			case 'id':
				return $item->id;

			case 'user_id':
				$user = bp_core_fetch_avatar( array( 'item_id' => $item->user_id, 'size' => 25 ) ) . '<br/>' .  bp_core_get_userlink( $item->user_id );
				return $user;

			case 'total_reports':
				return $item->total_reports_count;

			case 'current_reports':
				return $item->current_reports_count;

			case 'type':
				$reportable = bpmts_moderation_tools()->reportable_types->get( $item->item_type );
				return $reportable ? $reportable->label : $item->item_type;

			case 'updated_at':
				return mysql2date( 'g:i:s A, F j, Y', $item->updated_at );
		}
	}

	/**
	 * Get the item column.
	 *
	 * @param Queue $item queue item.
	 *
	 * @return string
	 */
	public function column_item( $item ) {
		$reportable = bpmts_moderation_tools()->reportable_types->get( $item->item_type );
		if ( ! $reportable ) {
			return __( 'Unavailable type', 'buddypress-moderation-tools' );
		}

		return sprintf( '<div class="bpmts-title-col-entry bpmts-type-{%s}-title-col-entry">%s</div>',$reportable->type, $reportable->get_item_title( $item->item_id ) );
	}

	/**
	 * Get the item column.
	 *
	 * @param Queue $item queue item.
	 *
	 * @return string
	 */
	public function column_thumbnail( $item ) {
		$reportable = bpmts_moderation_tools()->reportable_types->get( $item->item_type );
		if ( ! $reportable ) {
			return __( 'Unavailable type', 'buddypress-moderation-tools' );
		}

		return sprintf( '<div class="bpmts-thumb-col-entry bpmts-type-{%s}-thumb-col-entry">%s</div>',$reportable->type, $reportable->get_item_thumbnail( $item->item_id, 'thumb' ) );
	}

	/**
	 * Handle row actions.
	 *
	 * @param Queue  $item Queue item object.
	 * @param string $column_name column name.
	 * @param string $primary is primary column.
	 *
	 * @return bool|int|string
	 */
	public function handle_row_actions( $item, $column_name, $primary ) {
		$actions = array();

		switch ( $column_name ) {
			case 'item':
			case 'actions':
			case 'thumbnail':
				$url = add_query_arg(
					array(
						'queue_id' => $item->id,
						'page'     => 'bpmts-moderation',
					), admin_url( 'admin.php' )
				);

				$view_url   = add_query_arg( array( 'bpmts_action' => 'view-reports' ), $url );
				$delete_url = add_query_arg( array( 'bpmts_action' => 'delete-entry' ), $url );
				$delete_url = wp_nonce_url( $delete_url, 'delete-entry' );
				$actions    = array(
					'view'   => sprintf( '<a href="%s" title="%s">%s</a>', $view_url, __( 'View Reports', 'buddypress-moderation-tools' ), __( 'View', 'buddypress-moderation-tools' ) ),
					'delete' => sprintf( '<a href="%s" title="%s">%s</a>', $delete_url, __( 'Remove from Queue', 'buddypress-moderation-tools' ), __( 'Delete', 'buddypress-moderation-tools' ) ),
				);

				break;
		}

		return $this->row_actions( $actions );
	}

	/**
	 * Filter list by moderation state
	 *
	 * @param string $which Whether this is being invoked above ("top")
	 *                      or below the table ("bottom").
	 */
	protected function extra_tablenav( $which ) {
		if ( 'bottom' === $which ) {
			return;
		}

		$id        = 'bpmts_state';
		$button_id = 'filter';
		$is_hidden = empty( $this->args['hidden'] ) ? 0 : 1;

		?>
        <div class="alignleft actions">

            <select name="<?php echo $id ?>" id="<?php echo $id ?>">
                <option value="0" <?php selected( $is_hidden, 0, true ); ?>><?php _e( 'All Reported Items', 'buddypress-moderation-tools' ) ?></option>
                <option value="1" <?php selected( $is_hidden, 1, true ); ?>><?php _e( 'Hidden Reported Items', 'buddypress-moderation-tools' ) ?></option>
            </select>
			<?php submit_button( __( 'Filter', 'buddypress-moderation-tools' ), '', $button_id, false ); ?>
            <input type="hidden" name="bpmts_type" value="<?php echo esc_attr( $this->args['type'] );?>">
        </div>
		<?php
	}

}
