<?php
/**
 * Admin Loader.
 *
 * @package    BuddyPress Moderation Tools
 * @subpackage Admin
 * @copyright  Copyright (c) 2018, Brajesh Singh
 * @license    https://www.gnu.org/licenses/gpl.html GNU Public License
 * @author     Brajesh Singh
 * @since      1.0.0
 */

namespace PressThemes\ModerationTools\Admin;

use PressThemes\ModerationTools\Models\Log;
use PressThemes\ModerationTools\Models\Queue;
use PressThemes\ModerationTools\Models\Report;

/**
 * Queue manager
 *
 * @property-read string $bulk_action bulk action.
 * @property-read string $action moderation action.
 * @property-read int $queue_id queue id.
 * @property-read int $report_id queue id.
 */
class Queue_Manager {

	/**
	 * Args.
	 *
	 * @var array
	 */
	private $args = array();

	/**
	 * List table.
	 *
	 * @var \WP_List_Table
	 */
	private $table = null;

	/**
	 * Queue_Manager constructor.
	 *
	 * @param array $args args.
	 */
	public function __construct( $args = array() ) {
		$this->args = $args;
	}

	/**
	 * Magic getter.
	 *
	 * @param string $name name.
	 *
	 * @return mixed|null
	 */
	public function __isset( $name ) {
		return isset( $this->args[ $name ] ) ? true : false;
	}

	/**
	 * Magic getter.
	 *
	 * @param string $name name.
	 *
	 * @return mixed|null
	 */
	public function __get( $name ) {
		return isset( $this->args[ $name ] ) ? $this->args[ $name ] : null;
	}

	/**
	 * Get Table.
	 *
	 * @return \WP_List_Table
	 */
	public function get_table() {
		return $this->table;
	}

	/**
	 * Process Queue actions.
	 */
	public function process() {

		$this->maybe_redirect();

		if ( empty( $this->queue_id ) ) {
			$this->table = new Queue_Items_Table( $this->args );
		} elseif ( $this->report_id ) {

			// single report action.
			$this->handle_queue_reports_actions();
		} else {
			// view queue entry log.
			$this->handle_queue_entry_actions();
		}

	}

	/**
	 * Handle redirection to single entry view.
	 */
	private function maybe_redirect() {

		// first, let us check for the redirect?
		if ( empty( $this->queue_id ) && 'item-report-view' === $this->action && ! empty( $_GET['item_type'] ) && ! empty( $_GET['item_id'] ) ) {
			$queue = Queue::first( array(
				'item_type' => trim( $_GET['item_type'] ),
				'item_id'   => absint( $_GET['item_id'] ),
			) );

			if ( $queue ) {
				bpmts_redirect( bpmts_admin_get_queue_entry_url( array( 'queue_id' => $queue->id ) ) );
			}
		}
	}

	/**
	 * Handle queue entry actions.
	 */
	private function handle_queue_entry_actions() {

		$bulk_action = $this->bulk_action;

		if ( $bulk_action && 'delete' == $bulk_action ) {
			$done = $this->bulk_delete_reports();
			$args = $done ? array(
				'queue_id' => $this->queue_id,
				'message'  => __( 'Deleted', 'buddypress-moderation-tools' ),
			) : array(
				'queue_id'     => $this->queue_id,
				'message'      => __( 'Unable to perform the action!', 'buddypress-moderation-tools' ),
				'message-type' => 'error',
			);
			bpmts_redirect( bpmts_admin_get_queue_url( $args ) );
		}

		switch ( $this->action ) {

			case 'delete-entry':
				Queue::destroy( array( 'id' => $this->queue_id ) );
				Report::destroy( array( 'queue_id' => $this->queue_id ) );
				Log::destroy( array( 'queue_id' => $this->queue_id ) );

				$message = __( 'Item removed from queue.', 'buddypress-moderation-tools' );
				// redirect to the queue.
				bpmts_redirect( bpmts_admin_get_queue_url( array( 'message' => $message ) ) );
				break;
			case 'view-entry':
			default:
				$this->table = new Reports_Table( $this->args );
				break;

			case 'delete-item':
				if ( ! isset( $_GET['_wpnonce'] ) ) {
					return;// don't do anything.
				}

				if ( ! wp_verify_nonce( $_GET['_wpnonce'], 'delete-item' ) ) {
					bpmts_redirect( bpmts_admin_get_queue_entry_url( array(
						'queue_id'     => $this->queue_id,
						'message'      => __( 'Invalid action.', 'buddypress-moderation-tools' ),
						'message-type' => 'error',
					) ) );
				}

				$queue = Queue::find( $this->queue_id );
				if ( ! $queue ) {
					bpmts_redirect( bpmts_admin_get_queue_entry_url( array(
						'queue_id'     => $this->queue_id,
						'message'      => __( 'Invalid action.', 'buddypress-moderation-tools' ),
						'message-type' => 'error',
					) ) );
				}

				// if we are here, all is good.
				$deleted = bpmts_delete_item( $queue->item_id, $queue->item_type );
				if ( $deleted ) {
					bpmts_redirect( bpmts_admin_get_queue_url( array(
						'message' => __( 'Deleted successfully.', 'buddypress-moderation-tools' ),
					) ) );
				} else {
					bpmts_redirect( bpmts_admin_get_queue_url( array(
						'message'      => __( 'There was a problem. Please try again.', 'buddypress-moderation-tools' ),
						'message-type' => 'error',
					) ) );
				}

				break;

			case 'restore-item':
				if ( ! isset( $_GET['_wpnonce'] ) ) {
					return;// don't do anything.
				}

				if ( ! wp_verify_nonce( $_GET['_wpnonce'], 'restore-item' ) ) {
					bpmts_redirect( bpmts_admin_get_queue_entry_url( array(
						'queue_id'     => $this->queue_id,
						'message'      => __( 'Invalid action.', 'buddypress-moderation-tools' ),
						'message-type' => 'error',
					) ) );
				}

				$queue = Queue::find( $this->queue_id );
				if ( ! $queue ) {
					bpmts_redirect( bpmts_admin_get_queue_entry_url( array(
						'queue_id'     => $this->queue_id,
						'message'      => __( 'Invalid action.', 'buddypress-moderation-tools' ),
						'message-type' => 'error',
					) ) );
				}

				// if we are here, restore.
				$restored = bpmts_restore_item( $queue->item_id, $queue->item_type );
				if ( $restored ) {
					bpmts_redirect( bpmts_admin_get_queue_url( array(
						'message' => __( 'Restored successfully.', 'buddypress-moderation-tools' ),
					) ) );
				} else {
					bpmts_redirect( bpmts_admin_get_queue_url( array(
						'message'      => __( 'There was a problem. Please try again.', 'buddypress-moderation-tools' ),
						'message-type' => 'error',
					) ) );
				}

				break;

			case 'clean-item':
				if ( ! isset( $_GET['_wpnonce'] ) ) {
					return;// don't do anything.
				}

				if ( ! wp_verify_nonce( $_GET['_wpnonce'], 'clean-item' ) ) {
					bpmts_redirect( bpmts_admin_get_queue_entry_url( array(
						'queue_id'     => $this->queue_id,
						'message'      => __( 'Invalid action.', 'buddypress-moderation-tools' ),
						'message-type' => 'error',
					) ) );
				}

				$queue = Queue::find( $this->queue_id );
				if ( ! $queue ) {
					bpmts_redirect( bpmts_admin_get_queue_entry_url( array(
						'queue_id'     => $this->queue_id,
						'message'      => __( 'Invalid action.', 'buddypress-moderation-tools' ),
						'message-type' => 'error',
					) ) );
				}

				// if we are here, restore.
				$restored = bpmts_clean_item( $queue->item_id, $queue->item_type );
				if ( $restored ) {
					bpmts_redirect( bpmts_admin_get_queue_url( array(
						'message' => __( 'Marked item clean successfully.', 'buddypress-moderation-tools' ),
					) ) );
				} else {
					bpmts_redirect( bpmts_admin_get_queue_url( array(
						'message'      => __( 'There was a problem. Please try again.', 'buddypress-moderation-tools' ),
						'message-type' => 'error',
					) ) );
				}

				break;

		}
	}

	/**
	 * Handle queue reports for single report.
	 */
	private function handle_queue_reports_actions() {
		switch ( $this->action ) {

			case 'delete-report':
				Report::destroy( array( 'id' => $this->report_id ) );
				$message = __( 'Report deleted', 'buddypress-moderation-tools' );

				bpmts_redirect( bpmts_admin_get_queue_entry_url( array(
					'message'  => $message,
					'queue_id' => $this->queue_id,
				) ) );

				break;
		}

	}

	/**
	 * Bulk delete reports.
	 *
	 * @return bool
	 */
	private function bulk_delete_reports() {

		if ( ! isset( $_GET['_bpmtp_reports_nonce'] ) || ! wp_verify_nonce( $_GET['_bpmtp_reports_nonce'], 'bpmtp-reports-action' ) ) {
			return false;
		}

		$report_ids = isset( $_GET['items'] ) ? $_GET['items'] : array();

		if ( empty( $report_ids ) ) {
			return false;
		}

		Report::destroy( array( 'id' => array( 'op' => 'IN', 'value' => $report_ids ) ) );

		return true;
	}
}