<?php
/**
 * Report Abuse:- User functions.
 *
 * @package    ReportAbuse
 * @subpackage User
 * @copyright  Copyright (c) 2018, Brajesh Singh
 * @license    https://www.gnu.org/licenses/gpl.html GNU Public License
 * @author     Brajesh Singh
 * @since      1.0.0
 */

use PressThemes\ModerationTools\Schema\Schema;

/**
 * Is the given user hidden?
 *
 * @param int  $user_id user id.
 * @param bool $cache_only check in cache only.
 *
 * @return bool
 */
function bpmts_is_user_hidden( $user_id , $cache_only = false) {
	return bpmts_is_item_hidden( $user_id, 'user', $cache_only );
}
/**
 * Is the logged user hidden?
 *
 * @return bool
 */
function bpmts_is_loggedin_user_hidden() {
	if ( ! is_user_logged_in() ) {
		return false;
	}

	static $is_hidden;

	if ( isset( $is_hidden ) ) {
		return $is_hidden;
	}
	$is_hidden = bpmts_is_user_hidden( get_current_user_id() );

	return $is_hidden;
}

/**
 * Is user blocked from login.
 *
 * @param int $user_id user id.
 *
 * @return bool
 */
function bpmts_is_user_login_disabled( $user_id ) {
	return bpmts_get_option( 'user_disable_login' ) && bpmts_is_user_hidden( $user_id );
}

/**
 * Get all users who have exceeded the threshold.
 *
 * @return array
 */
function bpmts_get_all_hidden_user_ids() {
	global $wpdb;
	$table         = Schema::table( 'queue' );
	// get all hidden users.
	$users_sql = $wpdb->prepare( "SELECT item_id FROM {$table} WHERE item_type=%s AND is_hidden = %d", 'user', 1 );
	return $wpdb->get_col( $users_sql );
}

/**
 * Get all the users reported by the given user.
 *
 * @param int $user_id user id.
 *
 * @return array
 */
function bpmts_get_all_reported_users( $user_id ) {
	global $wpdb;
	$table         = Schema::table( 'queue' );
	$reports_table = Schema::table( 'reports' );
	// get all hidden users.
	$users_sql = $wpdb->prepare( "SELECT item_id FROM {$table} WHERE item_type=%s AND is_hidden = %d", 'user', 1 );

	$users_sql .= $wpdb->prepare( " AND id IN( SELECT queue_id FROM {$reports_table} WHERE reporter_id = %d)", $user_id );

	return $wpdb->get_col( $users_sql );
}

/**
 * Get all administrator.
 *
 * @return array
 */
function bpmts_get_administrators() {
	return get_users( array(
		'role' => 'administrator',
	) );
}

/**
 * Is the given user manager?
 *
 * @param int $user_id user id.
 *
 * @return bool
 */
function bpmts_is_user_manager( $user_id ) {
	return is_super_admin( $user_id );
}

/**
 * Destroy all sessions for the user.
 *
 * @param int $user_id user id.
 *
 * @return bool
 */
function bpmts_destroy_user_sessions( $user_id ) {
	$user = get_userdata( $user_id );
	if ( ! $user ) {
		return false;
	}

	$sessions = WP_Session_Tokens::get_instance( $user->ID );
	$sessions->destroy_all();

	return true;
}

/**
 * Check if the given user is same as logged user.
 *
 * @param int $user_id user id.
 *
 * @return bool
 */
function bpmts_is_me( $user_id ) {
	return get_current_user_id() == $user_id;
}
