<?php
/**
 * BuddyPress Moderation Tools:- Activity Comment Query filter.
 *
 * Note: it is not used as the comments are not filterable.
 *
 * @package    BuddyPress Moderation Tools
 * @subpackage Activity_Comment
 * @copyright  Copyright (c) 2018, Brajesh Singh
 * @license    https://www.gnu.org/licenses/gpl.html GNU Public License
 * @author     Brajesh Singh
 * @since      1.0.0
 */

namespace PressThemes\ModerationTools\Reportable_Types\BuddyPress\Activity_Comment\Filters;

use PressThemes\ModerationTools\Filters\Query_Filter;
use PressThemes\ModerationTools\Schema\Schema;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Filter Activity Comment Query.
 */
class Activity_Comment_Query_Filter extends Query_Filter {
	/**
	 * All comment ids prefetched.
	 *
	 * @var array
	 */
	private $comment_ids = array();

	/**
	 * Array of hidden ids.
	 *
	 * @var array
	 */
	private $hidden_ids = array();

	/**
	 * Boot this class.
	 */
	public static function boot() {
		$self = new self();
		$self->setup();
	}

	/**
	 * Setup hooks callbacks
	 */
	public function setup() {
		// for parent.
		$this->item_type = 'bp_acomment';

		$this->settings = array(
			'auto_hide' => bpmts_get_option( 'bp_acomment_auto_hide', 0 ),
			'hide_type' => bpmts_get_option( 'bp_acomment_hide_type', array( 'all' ) ),
		);

		$is_super_admin = is_super_admin();
		// each time prefetch is called, assume a new loop.
		add_filter( 'bp_activity_prefetch_object_data', array( $this, 'prefetch' ) );
		// let us prefetch the hidden ids too.
		add_filter( 'bp_activity_prefetch_object_data', array( $this, 'prefetch_hidden_ids' ) );

		if ( ! $is_super_admin ) {
			// since we don't have any other way, we will be doing it the incorrect way.
			// add_action( 'bp_activity_entry_content', array( $this, 'hide_comments' ) );
			add_action( 'bp_before_activity_entry_comments', array( $this, 'hide_comments' ) );
			add_filter( 'bp_activity_current_comment', array( $this, 'hide_comment_reply' ) );
		} else {
			// I am breaking the rule here, but this is the most efficient way.
			add_action( 'bp_activity_comment_options', array( $this, 'show_hidden_tag' ), 1003 );
		}
	}

	/**
	 * Prefetch and save the comment ids used in current loop.
	 *
	 * @param \BP_Activity_Activity[] $activities activities array.
	 *
	 * @return  \BP_Activity_Activity[] $activities activities array.
	 */
	public function prefetch( $activities ) {

		if ( empty( $activities ) || ! is_array( $activities ) ) {
			return $activities;
		}

		foreach ( $activities as $activity ) {
			if ( empty( $activity->children ) ) {
				continue;
			}

			$this->comment_ids = array_merge( $this->comment_ids, wp_list_pluck( $activity->children, 'id' ) );
			$this->prefetch( $activity->children );
		}
		return $activities;
	}

	/**
	 * Prefetch hidden ids.
	 *
	 * @param array $activities activities array.
	 *
	 * @return array
	 */
	public function prefetch_hidden_ids( $activities ) {

		if ( empty( $this->comment_ids ) ) {
			return $activities;
		}

		$hidden_ids = $this->get_hidden_ids( $this->comment_ids );

		$this->_cache_ids( $hidden_ids );

		foreach ( $hidden_ids as $id ) {
			$this->hidden_ids[ $id ] = $id;
		}

		return $activities;
	}

	/**
	 * Hide comments.
	 *
	 * @return bool|void
	 */
	public function hide_comments() {

		if ( empty( $this->hidden_ids ) || is_super_admin() ) {
			return;
		}

		global $activities_template;

		if ( empty( $activities_template ) || empty( $activities_template->activity ) || empty( $activities_template->activity->children ) ) {
			return;
		}

		// Remove comments if needed.
		foreach ( $activities_template->activity->children as $id => $child ) {
			if ( isset( $this->hidden_ids[ $child->id ] ) ) {
				unset( $activities_template->activity->children[ $id ] );
			}
		}

	}

	/**
	 * Hide comment replies.
	 *
	 * @param \BP_Activity_Activity $comment activity.
	 *
	 * @return mixed
	 */
	public function hide_comment_reply( $comment ) {
		// filter comment children.
		if ( is_super_admin() || empty( $comment ) || empty( $comment->children ) || empty( $this->hidden_ids ) ) {
			return $comment;
		}

		// Remove comments if needed.
		foreach ( $comment->children as $id => $child ) {
			if ( isset( $this->hidden_ids[ $child->id ] ) ) {
				unset( $comment->children[ $id ] );
			}
		}

		return $comment;
	}

	/**
	 * Show tag if needed.
	 */
	public function show_hidden_tag() {

		$comment_id = bp_get_activity_comment_id();

		if ( ! bpmts_is_item_hidden( $comment_id, $this->item_type, true ) ) {
			return;
		}

		echo bpmts_get_hidden_info_box( array(
			'item_type'   => 'bp_acomment',
			'item_id'     => $comment_id,
			'use_wrapper' => false,
			'link_class'  => 'bp-secondary-action',
		) );
	}

}
