<?php
/**
 * BuddyPress Activity Reportable support.
 *
 * @package    BuddyPress Moderation Tools
 * @subpackage Reportable
 * @copyright  Copyright (c) 2018, Brajesh Singh
 * @license    https://www.gnu.org/licenses/gpl.html GNU Public License
 * @author     Brajesh Singh
 * @since      1.0.0
 */

namespace PressThemes\ModerationTools\Reportable_Types\BuddyPress;

use Press_Themes\PT_Settings\Page;
use PressThemes\ModerationTools\Reportable_Types\BuddyPress\Activity\Admin\Activity_Settings_Panel;
use PressThemes\ModerationTools\Reportable_Types\BuddyPress\Activity\Filters\Activity_Query_Filter;
use PressThemes\ModerationTools\Reportable_Types\BuddyPress\Activity\Handlers\Activity_Action_Handler;
use PressThemes\ModerationTools\Reportable_Types\BuddyPress\Activity\Handlers\Activity_UI_Helper;
use PressThemes\ModerationTools\Reportable_Types\Reportable_Item_Type;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Reportable_Type_BP_Activity Activity
 */
class Reportable_Type_BP_Activity extends Reportable_Item_Type {

	/**
	 * Reportable_BP_Activity constructor.
	 */
	public function __construct() {
		parent::__construct( 'bp_activity', __( 'BuddyPress Activity', 'buddypress-moderation-tools' ) );
	}

	/**
	 * Load dependencies if any.
	 */
	public function boot() {
	}

	/**
	 * Setup.
	 */
	public function setup() {
		// do some setup.
		Activity_Query_Filter::boot();
		// Boot action handlers.
		Activity_Action_Handler::boot();
		// Boot UI Helper to add messages/buttons as needed.
		Activity_UI_Helper::boot();
	}

	/**
	 * Add activity admin settings panel.
	 *
	 * @param Page $page page object.
	 */
	public function add_admin_settings( $page ) {
		$panel = $page->add_panel( 'bp-activity-settings', __( 'Activity', 'buddypress-moderation-tools' ), __( 'BuddyPress Activity reporting setting', 'bp-report-activity' ) );
		Activity_Settings_Panel::build( $panel );
	}

	/**
	 * Delete an item of this type.
	 *
	 * @param int $item_id item id.
	 */
	public function delete( $item_id ) {
		bp_activity_delete_by_activity_id( $item_id );
	}

	/**
	 * Check that activity exists.
	 *
	 * @param int    $item_id post id.
	 * @param string $context ignore.
	 * @param int    $context_id ignore.
	 *
	 * @return bool
	 */
	public function is_valid( $item_id, $context, $context_id ) {
		$activity = new \BP_Activity_Activity( $item_id );
		return $activity && $activity->id;
	}

	/**
	 * Get user threshold.
	 *
	 * @return int
	 */
	public function get_threshold() {
		return bpmts_get_option( 'bp_activity_threshold', 5 );
	}

	/**
	 * Get the item object.
	 *
	 * @param int $item_id group id.
	 *
	 * @return false|\BP_Activity_Activity
	 */
	public function get_item( $item_id ) {
		return new \BP_Activity_Activity( $item_id );
	}

	/**
	 * Get the thumbnail
	 *
	 * @param int    $item_id item id.
	 * @param string $size thumb|full.
	 *
	 * @return string
	 */
	public function get_item_thumbnail( $item_id, $size ) {
		return '';
	}

	/**
	 * Get item title.
	 *
	 * @param int $item_id item id.
	 *
	 * @return string
	 */
	public function get_item_title( $item_id ) {
		$activity = $this->get_item( $item_id );

		return $activity? $activity->action : __( 'Invalid Activity.', 'buddypress-moderation-tools' );
	}

	/**
	 * Get item description.
	 *
	 * @param int $item_id item id.
	 *
	 * @return string
	 */
	public function get_item_description( $item_id ) {
		$activity = $this->get_item( $item_id );
		return $activity->id ? $activity->content : '';
	}

	/**
	 * Get Post Permalink.
	 *
	 * @param int $item_id user id.
	 *
	 * @return false|string
	 */
	public function get_item_permalink( $item_id ) {
		return bp_activity_get_permalink( $item_id );
	}

	/**
	 * Get item edit url.
	 *
	 * @param int $item_id item id.
	 *
	 * @return string
	 */
	public function get_item_edit_url( $item_id ) {
		return  bp_get_admin_url( 'admin.php?page=bp-activity&amp;aid=' . $item_id ) . '&amp;action=edit';
	}

	/**
	 * Get item delete url.
	 *
	 * @param int $item_id item id.
	 *
	 * @return string
	 */
	public function get_item_delete_url( $item_id ) {
		$spam_nonce = esc_html( '_wpnonce=' . wp_create_nonce( 'spam-activity_' . $item_id ) );

		return bp_get_admin_url( 'admin.php?page=bp-activity&amp;aid=' . $item_id ) . "&amp;action=delete&amp;$spam_nonce";
	}
}
