<?php
/**
 * BuddyPress Moderation Tools:- Group Query filter.
 *
 * @package    BuddyPress Moderation Tools
 * @subpackage Group
 * @copyright  Copyright (c) 2018, Brajesh Singh
 * @license    https://www.gnu.org/licenses/gpl.html GNU Public License
 * @author     Brajesh Singh
 * @since      1.0.0
 */

namespace PressThemes\ModerationTools\Reportable_Types\BuddyPress\Group\Filters;

use PressThemes\ModerationTools\Filters\Query_Filter;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Filter Group Query.
 */
class Group_Query_Filter extends Query_Filter {

	/**
	 * Boot this class.
	 */
	public static function boot() {
		$self = new self();
		$self->setup();
	}

	/**
	 * Setup hooks callbacks
	 */
	public function setup() {
		$this->item_type = 'bp_group';
		$this->settings = array(
			'auto_hide' => bpmts_get_option( 'bp_group_auto_hide', 0 ),
			'hide_type' => bpmts_get_option( 'bp_group_hide_type', array( 'all' ) ),
		);

		// hide from list everywhere.
		add_action( 'bp_groups_get_paged_groups_sql', array( $this, 'exclude_in_listing' ), 200, 3 );

		// filter total count.
		add_filter( 'bp_get_total_group_count', array( $this, 'fix_group_count' ) );

		add_filter( 'groups_get_groups', array( $this, 'prefetch' ) );
		// cache groups

	}

	public function prefetch( $details ) {

		$groups = $details['groups'];
		if ( empty( $groups ) ) {
			return $details;
		}

		$ids = wp_list_pluck( $groups, 'id' );
		$this->cache_hidden_items( $ids );

		return $details;
	}
	/**
	 * Exclude groups from query.
	 *
	 * @param string $paged_sql Concatenated SQL statement.
	 * @param array  $sql Array of SQL parts before concatenation.
	 * @param array  $r Array of parsed arguments for the get method.
	 *
	 * @return string
	 */
	public function exclude_in_listing( $paged_sql, $sql, $r ) {
		// do not hide users inside the admin.
		if ( is_super_admin() || ( is_admin() && ! defined( 'DOING_AJAX' ) ) ) {
			return $paged_sql;
		}

		$where_conditions = array();

		if ( $this->hide_for_reporters_enabled() ) {
			$where_conditions[] = "g.id NOT IN ({$this->get_sql_for_reported_items( get_current_user_id(), 'bp_group' )})";
		}

		if ( $this->hide_for_all_enabled() ) {
			$where_conditions[] = "g.id NOT IN ({$this->get_sql_for_hidden_items( 'bp_group' )})";
		}

		if ( empty( $where_conditions ) ) {
			return $paged_sql;
		}


		$where = '';
		if ( ! empty( $sql['where'] ) ) {
			$where = "WHERE {$sql['where']}";
		}

		if ( $where ) {
			$where .= ' AND ';
		}

		$where     .= join( ' AND ', $where_conditions );
		$paged_sql = "{$sql['select']} FROM {$sql['from']} {$where} {$sql['orderby']} {$sql['pagination']}";

		return $paged_sql;
	}

	/**
	 * Filter total active groups count and subtract the hidden group count
	 *
	 * @param int $count count.
	 *
	 * @return int
	 */
	public function fix_group_count( $count ) {
		return absint( $count - $this->get_hidden_groups_count() );
	}

	/**
	 * Get total no. of hidden groups.
	 *
	 * @return int
	 */
	public function get_hidden_groups_count() {
		global $wpdb;

		$where_conditions = array();

		if ( $this->hide_for_reporters_enabled() ) {
			$where_conditions[] = "g.id IN ({$this->get_sql_for_reported_items( get_current_user_id(), 'bp_group' )})";
		}

		if ( $this->hide_for_all_enabled() ) {
			$where_conditions[] = "g.id IN ({$this->get_sql_for_hidden_items( 'bp_group' )})";
		}

		if ( empty( $where_conditions ) ) {
			return 0;
		}

		$table     = buddypress()->groups->table_name;
		$where_sql = join( ' OR ', $where_conditions );

		$query = "SELECT COUNT(g.id) FROM {$table} as g WHERE {$where_sql}";

		return absint( $wpdb->get_var( $query ) );
	}
}
