<?php
/**
 * BuddyPress Moderation Tools:- Group Action handler.
 *
 * @package    BuddyPress Moderation Tools
 * @subpackage User
 * @copyright  Copyright (c) 2018, Brajesh Singh
 * @license    https://www.gnu.org/licenses/gpl.html GNU Public License
 * @author     Brajesh Singh
 * @since      1.0.0
 */

namespace PressThemes\ModerationTools\Reportable_Types\BuddyPress\Group\Handlers;


use PressThemes\ModerationTools\Messages\Message_Builder;
use PressThemes\ModerationTools\Models\Queue;
use PressThemes\ModerationTools\Models\Report;
use PressThemes\ModerationTools\Reportable_Types\BuddyPress\Group\Messages\Group_Message_Factory;

// Exit if files accessed directly over web.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Action handler.
 */
class Group_Action_Handler {

	/**
	 * Boot this class.
	 */
	public static function boot() {
		$self = new self();
		$self->setup();
	}

	/**
	 * Setup hooks callbacks
	 */
	public function setup() {

		add_action( 'bpmts_bp_group_reported', array( $this, 'on_report' ), 10, 3 );
		add_action( 'bpmts_bp_group_restored', array( $this, 'on_restore' ) );
		add_action( 'bpmts_bp_group_hidden', array( $this, 'on_hide' ), 10, 2 );

		// Notify before delete.
		add_action( 'bpmts_before_bp_group_delete', array( $this, 'on_delete' ), 10, 2 );

		// for access protection.
		if ( ! is_super_admin() ) {
			add_action( 'bp_template_redirect', array( $this, 'check_access' ) );
		}
	}

	/**
	 * Disable access to hidden groups? for everyone including group admin?
	 */
	public function check_access() {

		if ( ! bp_is_group() || ! bpmts_is_group_hidden( groups_get_current_group()->id ) ) {
			return;
		}

		bp_core_add_message( __( 'The group is unavailable currently. Please try again later.', 'buddypress-moderation-tolls' ), 'error' );

		$redirect_location = is_user_logged_in() ? bp_loggedin_user_domain() : home_url( '/' );
		// @todo add a filter here.
		bp_core_redirect( $redirect_location );
	}

	/**
	 * Notify admins via email on new report
	 *
	 * @param int    $group_id reported group id.
	 * @param Queue  $queue queue item.
	 * @param Report $report report object.
	 */
	public function on_report( $group_id, $queue, $report ) {

		// New report.
		if ( bpmts_get_option( 'bp_group_notify_admin_on_report' ) ) {
			$message_builder = Group_Message_Factory::build( 'admin-reported' );

			if ( ! $message_builder ) {
				return;
			}

			$message_builder->add_reports_token( $report );
			$this->notify_admin( $group_id, $message_builder );
		}
	}

	/**
	 * Notify user via email on account being hidden.
	 *
	 * @param int   $group_id id of group whose status has changed.
	 * @param Queue $queue queue entry.
	 */
	public function on_hide( $group_id, $queue ) {

		$is_super_admin = is_super_admin();

		// Group hidden, notify group admin.
		if ( bpmts_get_option( 'bp_group_notify_gadmin_on_hide' ) ) {
			$message_builder = Group_Message_Factory::build( 'user-hidden' );

			if ( ! $message_builder ) {
				return;
			}

			$this->notify_user( $group_id, $message_builder );
		}

		// Group hidden, notify site admin.
		if ( ! $is_super_admin && bpmts_get_option( 'bp_group_notify_admin_on_hide' ) ) {
			$message_builder = Group_Message_Factory::build( 'admin-hidden' );

			if ( ! $message_builder ) {
				return;
			}

			$message_builder->add_token_map( array( '{queue.entry}' => bpmts_admin_get_queue_entry_url( array( 'queue_id' => $queue->id ) ) ) );

			$this->notify_admin( $group_id, $message_builder );
		}
	}

	/**
	 * Notify user via email on account activation
	 *
	 * @param int $group_id id of group whose status has changed.
	 */
	public function on_restore( $group_id ) {

		// Account activated, send email to user.
		if ( bpmts_get_option( 'bp_group_notify_gadmin_on_restore' ) ) {
			$message_builder = Group_Message_Factory::build( 'user-restored' );

			if ( ! $message_builder ) {
				return;
			}
			$this->notify_user( $group_id, $message_builder );
		}
	}

	/**
	 * Notify group admins before delete.
	 *
	 * @param int              $group_id group id.
	 * @param \BP_Groups_Group $group grou object.
	 */
	public function on_delete( $group_id, $group ) {

		// Notify?
		if ( bpmts_get_option( 'bp_group_notify_gadmin_on_delete' ) ) {
			$message_builder = Group_Message_Factory::build( 'user-deleted' );

			if ( ! $message_builder ) {
				return;
			}
			$this->notify_user( $group_id, $message_builder );
		}
	}

	/**
	 * Notify admin on new report or hidden.
	 *
	 * @param int             $group_id id of group whose status has changed.
	 * @param Message_Builder $message_builder message builder.
	 */
	public function notify_admin( $group_id, $message_builder ) {

		$group          = new \BP_Groups_Group( $group_id );
		$administrators = bpmts_get_administrators();

		foreach ( $administrators as $administrator ) {
			$message = $message_builder->parse( $administrator, $group );
			if ( ! $message ) {
				continue;
			}

			wp_mail( $administrator->user_email, $message->subject, $message->message );
		}
	}

	/**
	 * Notify all group admin users via email on group hidden/restore.
	 *
	 * @param int             $group_id id of group whose status has changed.
	 * @param Message_Builder $message_builder message builder.
	 */
	public function notify_user( $group_id, $message_builder ) {

		$group = new \BP_Groups_Group( $group_id );

		$user_ids = wp_list_pluck( $group->admins, 'user_id' );
		foreach ( $user_ids as $user_id ) {
			$user = get_userdata( $user_id );
			if ( ! $user ) {
				continue;
			}
			$message = $message_builder->parse( $user, $group );
			wp_mail( $user->user_email, $message->subject, $message->message );
		}
	}
}

