<?php
/**
 * Post Type UI Helper.
 *
 * @package    BuddyPressModerationTools
 * @subpackage Geodirectory
 * @copyright  Copyright (c) 2018, Brajesh Singh
 * @license    https://www.gnu.org/licenses/gpl.html GNU Public License
 * @author     Brajesh Singh
 * @since      1.0.0
 */

/**
 * @contributor: Ravi Sharma(raviousprime)
 */

namespace PressThemes\ModerationTools\Reportable_Types\Geodirectory\Handlers;

// Exit if files access directly over web.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Buttons & Messages management.
 */
class Geodirectory_UI_Helper {

	/**
	 * Post type.
	 *
	 * @var string
	 */
	private $post_type = '';

	/**
	 * Boot this class.
	 *
	 * @param string $post_type post type.
	 */
	public static function boot( $post_type ) {
		$self            = new self();
		$self->post_type = $post_type;
		$self->setup();
	}

	/**
	 * Setup hooks callbacks
	 */
	public function setup() {
		// filter body.
		if ( is_user_logged_in() && is_super_admin() ) {
			add_filter( 'body_class', array( $this, 'add_body_class' ) );
			add_filter( 'geodir_post_view_extra_class', array( $this, 'add_list_class' ) );
		}

		// Boot UI Helper to add messages/buttons as needed.
		add_action( 'geodir_details_main_content', array( $this, 'add_report_buttons_geodir' ), 1001 );
		add_action( 'geodir_after_listing_post_excerpt', array( $this, 'add_report_buttons_geodir' ), 1001 );
	}

	/**
	 * Add appropriate class to body.
	 *
	 * @param array $classes css classes.
	 *
	 * @return array
	 */
	public function add_body_class( $classes ) {

		if ( ! is_user_logged_in() ) {
			return $classes;
		}

		if ( is_singular( $this->post_type ) ) {
			$classes[] = bpmts_is_item_hidden( get_queried_object_id(), $this->post_type ) ? "bpmts-displayed-item-hidden bpmts-displayed-{$this->post_type}-hidden" : "bpmts-displayed-item-clean bpmts-displayed-{$this->post_type}-clean";
		}

		return $classes;
	}

	/**
	 * Add extra class to member list entry.
	 *
	 * @return string
	 */
	public function add_list_class() {
		$classes = bpmts_is_item_hidden( get_the_ID(), $this->post_type ) ? "bpmts-item-entry-hidden bpmts-post-type-entry-{$this->post_type}-hidden" : "bpmts-item-entry-clean bpmts-post-type-entry-{$this->post_type}-clean";

		return $classes;
	}

	/**
	 * Add report button to geo-directory content.
	 *
	 * @param \WP_Post $post Post object.
	 */
	public function add_report_buttons_geodir( $post ) {
		if ( $post->post_type === $this->post_type && bpmts_user_can_report() ) {
			bpmts_report_button( array(
				'item_id'    => get_the_ID(),
				'item_type'  => $this->post_type,
				'context'    => $this->post_type,
				'context_id' => get_the_ID(),
			) );
		}
	}
}
