<?php
/**
 * Reportable support for User
 *
 * @package    BuddyPress Moderation Tools
 * @subpackage User
 * @copyright  Copyright (c) 2018, Brajesh Singh
 * @license    https://www.gnu.org/licenses/gpl.html GNU Public License
 * @author     Brajesh Singh
 * @since      1.0.0
 */

namespace PressThemes\ModerationTools\Reportable_Types\User;

use Press_Themes\PT_Settings\Page;
use PressThemes\ModerationTools\Reportable_Types\Reportable_Item_Type;
use PressThemes\ModerationTools\Reportable_Types\User\Admin\User_Settings_Panel;
use PressThemes\ModerationTools\Reportable_Types\User\Filters\User_Query_Filter;
use PressThemes\ModerationTools\Reportable_Types\User\Gate\User_Activity_Gate;
use PressThemes\ModerationTools\Reportable_Types\User\Gate\User_Gate;
use PressThemes\ModerationTools\Reportable_Types\User\Gate\User_Message_Gate;
use PressThemes\ModerationTools\Reportable_Types\User\Handlers\User_Action_Handler;
use PressThemes\ModerationTools\Reportable_Types\User\Handlers\User_UI_Helper;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Reportable user type
 */
class Reportable_Type_User extends Reportable_Item_Type {

	/**
	 * Reportable User constructor.
	 */
	public function __construct() {
		parent::__construct( 'user', __( 'User', 'buddypress-moderation-tools' ) );
	}

	/**
	 * Boot this module.
	 */
	public function boot() {
		// Nothing to do.
	}

	/**
	 * Setup hooks. Called after boot.
	 */
	public function setup() {
		// do some setup.
		// Boot action handlers.
		User_Action_Handler::boot();
		User_Query_Filter::boot();
		// Boot UI Helper to add messages/buttons as needed.
		User_UI_Helper::boot();

		// Restrictions.
		User_Activity_Gate::boot();
		User_Message_Gate::boot();
		User_Gate::boot();
	}

	/**
	 * Add User Settings.
	 *
	 * @param Page $page page object.
	 */
	public function add_admin_settings( $page ) {

		$panel = $page->add_panel( 'user', _x( 'User', 'Admin settings panel title', 'buddypress-moderation-tools' ) );

		User_Settings_Panel::build( $panel );

		if ( function_exists( 'bp_is_active' ) && bp_is_active( 'messages' ) ) {
		//	$panel = $page->add_panel( 'bp_message', _x( 'Private Message', 'Admin settings panel title', 'buddypress-moderation-tools' ) );
		//	Message_Settings_Panel::build( $panel );
		}
	}

	/**
	 * Delete an item of this type.
	 *
	 * @param int $item_id item id.
	 */
	public function delete( $item_id ) {
		wp_delete_user( $item_id );
	}

	/**
	 * Check that the given user exists.
	 *
	 * @param int    $item_id user id.
	 * @param string $context ignore.
	 * @param int    $context_id ignore.
	 *
	 * @return bool
	 */
	public function is_valid( $item_id, $context, $context_id ) {
		$user = get_user_by( 'id', $item_id );

		return $user && $user->ID;
	}

	/**
	 * Get user threshold.
	 *
	 * @return int
	 */
	public function get_threshold() {
		return bpmts_get_option( 'user_threshold', 5 );
	}

	/**
	 * Get the item object.
	 *
	 * @param int $item_id user id.
	 *
	 * @return false|\WP_User
	 */
	public function get_item( $item_id ) {
		return $user = get_user_by( 'id', $item_id );
	}

	/**
	 * Get the thumbnail
	 *
	 * @param int    $item_id item id.
	 * @param string $size thumb|full.
	 *
	 * @return string
	 */
	public function get_item_thumbnail( $item_id, $size ) {

		if ( ! $item_id ) {
			return 'Unavailable';
		}

		return get_avatar( $item_id, $size );
	}

	/**
	 * Get item title.
	 *
	 * @param int $item_id item id.
	 *
	 * @return string
	 */
	public function get_item_title( $item_id ) {
		$user = $this->get_item( $item_id );

		return $user? $user->display_name : __( 'Invalid User.', 'buddypress-moderation-tools' );
	}

	/**
	 * Get item description.
	 *
	 * @param int $item_id item id.
	 *
	 * @return string
	 */
	public function get_item_description( $item_id ) {
		$user = $this->get_item( $item_id );

		return $user ? $user->bio : '';
	}

	/**
	 * Get Post Permalink.
	 *
	 * @param int $item_id user id.
	 *
	 * @return false|string
	 */
	public function get_item_permalink( $item_id ) {
		if ( function_exists( 'buddypress' ) ) {
			return bp_core_get_user_domain( $item_id );
		} else {
			return get_edit_user_link( $item_id );
		}
	}

	/**
	 * Get item edit url.
	 *
	 * @param int $item_id item id.
	 *
	 * @return string
	 */
	public function get_item_edit_url( $item_id ) {
			return get_edit_user_link( $item_id );
	}

	/**
	 * Get item delete url.
	 *
	 * @param int $item_id item id.
	 *
	 * @return string
	 */
	public function get_item_delete_url( $item_id ) {
		return is_multisite() ? wp_nonce_url( "users.php?action=remove&amp;user=$item_id", 'bulk-users' ) : wp_nonce_url( "users.php?action=delete&amp;user=$item_id", 'bulk-users' );
	}
}
