<?php
/**
 * Category Manager
 *
 * @package    BuddyPress Moderation Tools
 * @subpackage Admin
 * @copyright  Copyright (c) 2018, Brajesh Singh
 * @license    https://www.gnu.org/licenses/gpl.html GNU Public License
 * @author     Brajesh Singh
 * @since      1.0.0
 */

namespace PressThemes\ModerationTools\Admin;

use PressThemes\ModerationTools\Models\Category;

/**
 * Admin category Manager
 */
class Category_Manager {

	/**
	 * Current Action.
	 *
	 * @var string
	 */
	private $action = '';

	/**
	 * Category id.
	 *
	 * @var int
	 */
	private $cat_id = 0;

	/**
	 * Handle actions.
	 */
	public function handle() {
		$this->parse_vars();

		if ( ! $this->cat_id ) {
			$this->handle_create();
		} elseif ( 'edit' === $this->action ) {
			$this->handle_update();
		} elseif ( 'delete' === $this->action ) {
			$this->handle_delete();
		}
	}

	/**
	 * Render View.
	 */
	public function render() {
		$this->parse_vars();

		$base = dirname( __FILE__ );

		if ( ! $this->cat_id ) {
			require $base . '/templates/category/home.php';
		} elseif ( 'edit' === $this->action ) {
			require $base . '/templates/category/edit.php';
		}
	}

	/**
	 * Handle category creation.
	 */
	private function handle_create() {

		if ( empty( $_POST['name'] ) || ! trim( $_POST['name'] ) ) {
			return;
		}

		if ( ! wp_verify_nonce( $_POST['_wpnonce'], 'bpmts_add_category' ) ) {
			wp_die( __( 'Invalid action.', 'buddypress-moderation-tools' ) );
		}

		// if we are here, create
		$args = array(
			'name'   => trim( $_POST['name'] ),
			'is_active' => empty( $_POST['is_active'] ) ? 0 : 1
		);

		$category = Category::create( $args );

		if ( $category->save() ) {
			$message = __( 'Successfully created.', 'buddypress-moderation-tools' );
			wp_cache_delete( 'active_categories', 'bpmts');
		} else {
			$message = __( 'There was a problem. Please try again later.', 'buddypress-moderation-tools' );
		}

		bpmts_redirect( $this->url( array( 'message' => $message ) ) );
	}

	/**
	 * Handle category update.
	 */
	private function handle_update() {

		if ( empty( $_POST['name'] ) || empty( $_POST['id'] ) || ! trim( $_POST['name'] ) ) {
			return;
		}

		if ( ! wp_verify_nonce( $_POST['_wpnonce'], 'bpmts_edit_category' ) ) {
			wp_die( __( 'Invalid action.', 'buddypress-moderation-tools' ) );
		}

		$args = array(
			'name'   => trim( $_POST['name'] ),
			'is_active' => empty( $_POST['is_active'] ) ? 0 : 1
		);

		if ( Category::update( $args , array( 'id' => $this->cat_id ) ) ) {
			$message = __( 'Updated successfully', 'buddypress-moderation-tools' );
			wp_cache_delete( 'active_categories', 'bpmts');
		} else {
			$message = __( 'There was a problem. Please try again later.', 'buddypress-moderation-tools' );
		}

		bpmts_redirect( $this->get_entry_url(
			array(
				'bpmts_action' => 'edit',
				'message'   => $message,
			)
		) );

	}


	/**
	 * Handle category delete.
	 */
	private function handle_delete() {

		if ( ! wp_verify_nonce( $_GET['_wpnonce'], 'bpmts_delete_category' ) ) {
			bpmts_redirect( $this->url() );
		}

		if ( Category::destroy( array( 'id' => $this->cat_id ) ) ) {
			$message = __( 'Category deleted successfully.', 'buddypress-moderation-tools' );
			wp_cache_delete( 'active_categories', 'bpmts');
		} else {
			$message = __( 'There was a problem deleting category. Please try again later.', 'buddypress-moderation-tools' );
		}

		bpmts_redirect( $this->url( array(
			'message' => $message,
		) ) );

	}

	/**
	 * Parse and setup vars.
	 */
	private function parse_vars() {
		$this->action = isset( $_GET['bpmts_action'] ) ? trim( $_GET['bpmts_action'] ) : '';
		$this->cat_id = isset( $_GET['bpmts_cat_id'] ) ? absint( $_GET['bpmts_cat_id'] ) : 0;
	}

	/**
	 * Get url for the queue item pages.
	 *
	 * @param array $args args.
	 *
	 * @return string
	 */
	private function url( $args = array() ) {

		$defaults = array(
			'page' => 'bpmts-category',
		);

		$args = wp_parse_args( $args, $defaults );
		$args = urlencode_deep( $args );

		return add_query_arg( $args, admin_url( 'admin.php' ) );
	}

	/**
	 * Get url for the queue item pages.
	 *
	 * @param array $args args.
	 *
	 * @return string
	 */
	private function get_entry_url( $args = array() ) {
		$defaults = array(
			'bpmts_cat_id' => $this->cat_id,
		);

		$args = wp_parse_args( $args, $defaults );

		return $this->url( $args );
	}
}