<?php
/**
 * Reports for a single item.
 *
 * @package    BuddyPress Moderation Tools
 * @subpackage Admin
 * @copyright  Copyright (c) 2018, Brajesh Singh
 * @license    https://www.gnu.org/licenses/gpl.html GNU Public License
 * @author     Brajesh Singh
 * @since      1.0.0
 */

namespace PressThemes\ModerationTools\Admin;

use PressThemes\ModerationTools\Models\Category;
use PressThemes\ModerationTools\Models\Report;

// Do not allow direct access over web.
defined( 'ABSPATH' ) || exit;

/**
 * Class Queue_Items_Table
 */
class Reports_Table extends \WP_List_Table {

	/**
	 * Flag vars.
	 *
	 * @var array
	 */
	private $args = array();

	/**
	 * Reports_Table constructor.
	 *
	 * @param array $args Array of values.
	 */
	public function __construct( $args = array() ) {
		$this->args  = $args;
		$parent_args = array(
			'singular' => 'report',
			'plural'   => 'reports',
			'ajax'     => false,
		);

		parent::__construct( $parent_args );
	}

	/**
	 * Check user permissions
	 *
	 * @return bool
	 */
	public function ajax_user_can() {
		return current_user_can( 'manage_options' );
	}

	/**
	 * All logic goes here
	 */
	public function prepare_items() {
		$current_page = $this->get_pagenum();
		$queue_id     = isset( $_GET['queue_id'] ) ? absint( $_GET['queue_id'] ) : 0;
		$log_id       = isset( $_GET['log_id'] ) ? absint( $_GET['log_id'] ) : 0;

		$args = array(
			'per_page' => 20,
			'page'     => $current_page,
		);

		if ( $log_id ) {
			$args['log_id'] = $log_id;
		} else {
			$args['queue_id'] = $queue_id;
		}

		$this->items = Report::get( $args );

		$this->set_pagination_args( array(
			'total_items' => count( $this->items ),
			'per_page'    => 20,
		) );
	}

	/**
	 * Render message when no items found
	 */
	public function no_items() {
		_e( 'No items in moderation.', 'buddypress-moderation-tools' );
	}

	/**
	 * Get views
	 *
	 * @return array
	 */
	protected function get_views() {

		$views = array();

		return $views;
	}

	/**
	 * Return bulk actions
	 *
	 * @return array
	 */
	protected function get_bulk_actions() {
		$bulk_actions = array(
			'delete' => __( 'Delete', 'buddypress-moderation-tools' ),
		);

		return $bulk_actions;
	}

	/**
	 * Get column info
	 *
	 * @return array
	 */
	public function get_columns() {
		$columns = array(
			'cb'          => '<input type="checkbox" />',
			'id'          => __( 'ID', 'buddypress-moderation-tools' ),
			'reporter_id' => __( 'Reporter', 'buddypress-moderation-tools' ),
		);

		$form_fields = bpmts_get_option( 'form_fields', array() );

		if ( in_array( 'email', $form_fields, true ) ) {
			$columns['reporter_email'] = __( 'Reporter Email', 'buddypress-moderation-tools' );
		}

		if ( in_array( 'category', $form_fields, true ) ) {
			$columns['category'] = __( 'Category', 'buddypress-moderation-tools' );

		}

		if ( in_array( 'subject', $form_fields, true ) ) {
			$columns['subject'] = __( 'Subject', 'buddypress-moderation-tools' );

		}

		if ( in_array( 'message', $form_fields, true ) ) {
			$columns['message'] = __( 'Message', 'buddypress-moderation-tools' );
		}

		$columns['created_at'] = __( 'Date', 'buddypress-moderation-tools' );

		return $columns;
	}

	/**
	 * Sortable columns.
	 *
	 * @return array
	 */
	public function get_sortable_columns() {

		$sortable_columns = array(
			'reporter_id' => array( 'reporter_id', false ),
			'category'    => array( 'category', false ),
		);

		return $sortable_columns;
	}

	/**
	 * Mask action
	 *
	 * @param mixed $item Current queue id.
	 *
	 * @return string
	 */
	public function column_cb( $item ) {
		return sprintf(
			'<input type="checkbox" name="items[]" value="%d" />', $item->id
		);
	}

	/**
	 * Column data.
	 *
	 * @param Report $item report item.
	 * @param string $col column name.
	 *
	 * @return string
	 */
	public function column_default( $item, $col ) {

		switch ( $col ) {

			case 'id':
				return $item->id;

			case 'reporter_email':
				return $item->email;

			case 'category':
				return Category::name( $item->category );

			case 'subject':
				return wp_kses_data( $item->subject );
			case 'message':
				return wp_kses_data( $item->message );

			case 'created_at':
				return mysql2date( 'g:i:s A, F j, Y', $item->created_at );
		}
	}

	/**
	 * Get the column data for reporter.
	 *
	 * @param Report $item report object.
	 *
	 * @return string
	 */
	public function column_reporter_id( $item ) {
		if ( ! $item->reporter_id ) {
			return $item->name;
		}

		$user = bp_core_fetch_avatar( array( 'item_id' => $item->reporter_id, 'size' => 25 ) ) . '<br/>' . bp_core_get_userlink( $item->reporter_id );
		return $user;
	}

	/**
	 * Handle row actions.
	 *
	 * @param Report $item report object.
	 * @param string $column_name column name.
	 * @param string $primary is primary column.
	 *
	 * @return bool|int|string
	 */
	public function handle_row_actions( $item, $column_name, $primary ) {
		$actions = array();

		switch ( $column_name ) {
			case 'id':
			case 'actions':
				$args = array(
					'queue_id'     => $item->queue_id,
					'report_id'    => $item->id,
					'bpmts_action' => 'delete-report',
					'page'         => 'bpmts-moderation',
				);

				if ( $this->args['log_id'] ) {
					$args['log_id'] = $this->args['log_id'];
					$args['page']   = 'bpmts-log';
				}
				$delete_url = add_query_arg( $args, admin_url( 'admin.php' ) );

				$actions = array(
					'delete' => sprintf( '<a href="%s">%s</a>', $delete_url, __( 'Delete', 'buddypress-moderation-tools' ) ),
				);

				break;

			case 'category':
				break;

			case 'date_updated':
				return mysql2date( 'g:i:s A, F j, Y', $item->updated_at );
				break;
		}

		return $this->row_actions( $actions );
	}

}
