<?php
/**
 * Bootstrapper. Initializes the plugin.
 *
 * @package    ReportAbuse
 * @subpackage Bootstrap
 * @copyright  Copyright (c) 2018, Brajesh Singh
 * @license    https://www.gnu.org/licenses/gpl.html GNU Public License
 * @author     Brajesh Singh
 * @since      1.0.0
 */

namespace PressThemes\ModerationTools\Bootstrap;

use PressThemes\ModerationTools\Admin\Admin_Loader;
use PressThemes\ModerationTools\Handlers\Ajax_Request_Handler;
use PressThemes\ModerationTools\Reportable_Types\BuddyPress\Reportable_Type_BP_Activity;
use PressThemes\ModerationTools\Reportable_Types\BuddyPress\Reportable_Type_BP_Activity_Comment;
use PressThemes\ModerationTools\Reportable_Types\BuddyPress\Reportable_Type_BP_Group;
use PressThemes\ModerationTools\Reportable_Types\BBPress\Reportable_Type_BBP_Forum;
use PressThemes\ModerationTools\Reportable_Types\BBPress\Reportable_Type_BBP_Reply;
use PressThemes\ModerationTools\Reportable_Types\BBPress\Reportable_Type_BBP_Topic;
use PressThemes\ModerationTools\Reportable_Types\Comment_Type\Reportable_Comment_Type;
use PressThemes\ModerationTools\Reportable_Types\MediaPress\Reportable_Type_MPP_Gallery;
use PressThemes\ModerationTools\Reportable_Types\MediaPress\Reportable_Type_MPP_Media;
use PressThemes\ModerationTools\Reportable_Types\Post_Type\Filters\Post_Type_Query_Filter;
use PressThemes\ModerationTools\Reportable_Types\Post_Type\Reportable_Post_Type;
use PressThemes\ModerationTools\Reportable_Types\Geodirectory\Reportable_Type_Geodirectory;
use PressThemes\ModerationTools\Reportable_Types\User\Reportable_Type_User;

// No direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit( 0 );
}

/**
 * Bootstrapper.
 */
class Bootstrapper {

	/**
	 * Setup the bootstrapper.
	 */
	public static function boot() {
		$self = new self();
		$self->setup();
	}

	/**
	 * Bind hooks
	 */
	private function setup() {
		// called after pp_loaded.
		add_action( 'plugins_loaded', array( $this, 'load' ) );
		add_action( 'plugins_loaded', array( $this, 'admin_load' ), 9996 ); // pt-settings 1.0.3.
		add_action( 'init', array( $this, 'load_translations' ) );
		add_action( 'init', array( $this, 'add_cache_group' ) );

		add_action( 'wp_loaded', array( $this, 'register_reportable_types' ), 1 );
		add_action( 'wp_loaded', array( $this, 'boot_reportable_types' ), 2 );
		add_action( 'wp_loaded', array( $this, 'setup_reportable_types' ), 3 );

		Assets_Loader::boot();
	}

	/**
	 * Load core functions/template tags.
	 * These are non auto loadable constructs.
	 */
	public function load() {
		$this->load_common();
		$this->load_admin();
	}
	/**
	 * Load admin.
	 */
	public function admin_load() {

		if ( ! is_admin() ) {
			return;
		}

		$files = array();

		if ( ! defined( 'DOING_AJAX' ) ) {
			$files[] = 'src/admin/pt-settings/pt-settings-loader.php';
		}

		$path = bpmts_moderation_tools()->path;
		foreach ( $files as $file ) {
			require_once $path . $file;
		}
	}

	/**
	 * Load translations.
	 */
	public function load_translations() {
		load_plugin_textdomain( 'buddypress-moderation-tools', false, basename( bpmts_moderation_tools()->path ) . '/languages' );
	}

	/**
	 * Add global cache group.
	 */
	public function add_cache_group() {
		wp_cache_add_global_groups( 'bpmts' );
	}

	/**
	 * Register supported item type.
	 */
	public function register_reportable_types() {
		$this->register_core_types();
		$this->register_bp_types();
		$this->register_mpp_types();
		$this->register_bbp_types();
		$this->register_geodir_types();
		/**
		 * Register custom reportable type son this action.
		 */
		do_action( 'bpmts_reportable_types_registered' );
	}

	/**
	 * Boot the reportable types.
	 */
	public function boot_reportable_types() {
		bpmts_moderation_tools()->reportable_types->boot();

		do_action( 'bpmts_reportable_types_booted' );
	}

	/**
	 * Let the reportable types setup themselves.
	 */
	public function setup_reportable_types() {
		bpmts_moderation_tools()->reportable_types->setup();

		do_action( 'bpmts_reportable_types_setup' );
	}


	/**
	 * Register core reportable types(post type, comment, user).
	 */
	private function register_core_types() {
		$post_types = get_post_types( array( 'public'=> true, '_builtin'=> false ) );

		if ( function_exists( 'geodir_get_posttypes' ) ) {
			// Geodir will have specific handling later.
			$excluded_post_types = geodir_get_posttypes();
			if ( is_array( $excluded_post_types ) ) {
				$post_types = array_diff( $post_types, $excluded_post_types );
			}
		}

		$post_types = apply_filters( 'bpmts_allowed_post_types', array_merge( array( 'post' ), $post_types ) );

		// Register all post types.
		foreach ( $post_types as $post_type ) {
			bpmts_moderation_tools()->reportable_types->add( new Reportable_Post_Type( $post_type ) );
		}

		// comments? Next version.
		// bpmts_moderation_tools()->reportable_types->add( new Reportable_Comment_Type( 'comment', __( 'Comment', 'buddypress-moderation-tools' ) ) );
		// User type.
		bpmts_moderation_tools()->reportable_types->add( new Reportable_Type_User() );
	}

	/**
	 * Register BuddyPress support.
	 */
	private function register_bp_types() {

		if ( ! function_exists( 'buddypress' ) ) {
			return;
		}

		if ( bp_is_active( 'activity' ) ) {
			bpmts_moderation_tools()->reportable_types->add( new Reportable_Type_BP_Activity() );
			bpmts_moderation_tools()->reportable_types->add( new Reportable_Type_BP_Activity_Comment() );
		}

		if ( bp_is_active( 'groups' ) ) {
			bpmts_moderation_tools()->reportable_types->add( new Reportable_Type_BP_Group() );
		}
	}

	/**
	 * Register BuddyPress support.
	 */
	private function register_bbp_types() {

		if ( ! function_exists( 'bbpress' ) ) {
			return;
		}

		bpmts_moderation_tools()->reportable_types->add( new Reportable_Type_BBP_Topic() );
		//bpmts_moderation_tools()->reportable_types->add( new Reportable_Type_BBP_Forum() );
		bpmts_moderation_tools()->reportable_types->add( new Reportable_Type_BBP_Reply() );
	}

	/**
	 * Register geodir places support.
	 */
	public function register_geodir_types() {
		global $wpdb;

		if ( ! function_exists( 'geodir_get_posttypes' ) ) {
			return;
		}

		$post_types = geodir_get_posttypes( 'options' );

		// Return if empty.
		if ( empty( $post_types ) ) {
			return;
		}

		foreach ( $post_types as $post_type => $label ) {

			if ( ! post_type_exists( $post_type ) ) {
				continue;
			}

			bpmts_moderation_tools()->reportable_types->add( new Reportable_Type_Geodirectory( $post_type ) );

			$query_filter = new Post_Type_Query_Filter();
			// Look for a way to improve it in future.
			$sql  = $query_filter->get_sql_for_hidden_items( $post_type );


			add_action(
				'pre_get_posts',
				function ( \WP_Query $query ) use ( $sql ) {

					if ( ! geodir_is_geodir_page() || is_super_admin() ) {
						return;
					}
					global $wpdb;
					$place_ids = $wpdb->get_col( $sql );

					if ( empty( $place_ids ) ) {
						return;
					}

					$query->set( 'post__not_in', $place_ids );
				}, 110
			);

			add_filter( 'geodir_filter_widget_listings_where', function ( $where ) use ( $sql ) {

				if ( is_super_admin() ) {
					return $where;
				}
				global $wpdb;
				$place_ids = $wpdb->get_col( $sql );

				if ( empty( $place_ids ) ) {
					return;
				}

				$post__not_in = implode( ',', array_map( 'absint', $place_ids ) );

				$where .= " AND wp_posts.ID NOT IN ($post__not_in)";

				return $where;
			}, 11 );
		}
	}

	/**
	 * Register support for MediaPress Gallery/media.
	 */
	private function register_mpp_types() {
		return; // Will be added in 1.1.

		if ( ! function_exists( 'mediapress' ) ) {
			return;
		}
		// add media and gallery support.
		bpmts_moderation_tools()->reportable_types->add( new Reportable_Type_MPP_Media() );
		bpmts_moderation_tools()->reportable_types->add( new Reportable_Type_MPP_Gallery() );
	}

	/**
	 * Load files common to each request type.
	 */
	private function load_common() {
		$path = bpmts_moderation_tools()->path;

		$files = array(
			'src/core/bpmts-functions.php',
			'src/core/bpmts-user-functions.php',
			'src/core/bpmts-template.php',
			'src/admin/bpmts-admin-functions.php',
		);

		foreach ( $files as $file ) {
			require_once $path . $file;
		}

		Ajax_Request_Handler::boot();
	}

	/**
	 * Load admin.
	 */
	private function load_admin() {
		if ( ! is_admin() ) {
			return;
		}

		// $path = ptreport_abuse()->path;

		if ( ! defined( 'DOING_AJAX' ) ) {
			Admin_Loader::boot();
		}
	}

}
