<?php
/**
 * BuddyPress Moderation Tools:- User Query filter.
 *
 * @package    BuddyPress Moderation Tools
 * @subpackage User
 * @copyright  Copyright (c) 2018, Brajesh Singh
 * @license    https://www.gnu.org/licenses/gpl.html GNU Public License
 * @author     Brajesh Singh
 * @since      1.0.0
 */

namespace PressThemes\ModerationTools\Filters;

use PressThemes\ModerationTools\Schema\Schema;

// Do not allow direct access over web.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Filter User Query.
 */
abstract class Query_Filter {

	/**
	 * Settings.
	 *
	 * @var array
	 */
	protected $settings = array();

	/**
	 * Item type.
	 *
	 * @var string
	 */
	protected $item_type = '';

	/**
	 * Setup hooks callbacks
	 */
	abstract public function setup();

	/**
	 * Get hidden sql for all items.
	 *
	 * @param string $item_type item type.
	 *
	 * @return string
	 */
	public function get_sql_for_hidden_items( $item_type ) {

		global $wpdb;

		$table_queue = Schema::table( 'queue' );

		// get all hidden items.
		return $wpdb->prepare( "SELECT item_id FROM {$table_queue} WHERE item_type= %s AND is_hidden = %d", $item_type, 1 );
	}
	/**
	 * Get sql clause for hiding the items from reporter.
	 *
	 * @param int    $reporter_id reporter id.
	 * @param string $item_type item type.
	 *
	 * @return string|void
	 */
	public function get_sql_for_reported_items( $reporter_id, $item_type ) {
		global $wpdb;
		$table_queue   = Schema::table( 'queue' );
		$table_reports = Schema::table( 'reports' );

		return $wpdb->prepare( " SELECT item_id FROM {$table_queue} WHERE item_type = %s AND is_new=%d AND id IN ( SELECT queue_id FROM {$table_reports} WHERE reporter_id = %d)", $item_type, 1, $reporter_id );
	}

	/**
	 * Is hide for all members enabled?
	 *
	 * @return bool
	 */
	public function hide_for_all_enabled() {
		return isset( $this->settings['auto_hide'] ) && is_array( $this->settings['hide_type'] ) && in_array( 'all', $this->settings['hide_type'], true );
	}

	/**
	 * Should we hide for reporter.
	 *
	 * @return bool
	 */
	public function hide_for_reporters_enabled() {
		return is_user_logged_in() && is_array( $this->settings['hide_type'] ) && in_array( 'reporter', $this->settings['hide_type'], true );
	}

	/**
	 * Get exclude activity sql.
	 *
	 * @param array $item_ids item ids.
	 *
	 * @return array
	 */
	protected function get_hidden_ids( $item_ids ) {
		global $wpdb;

		$table = Schema::table( 'queue' );

		$list = join( ',', $item_ids );
		// get all hidden ids.
		$where_conditions = array();
		// @todo rethink.
		$comment_ids_sql = $wpdb->prepare( "SELECT q.item_id FROM {$table} as q WHERE q.item_type=%s AND q.item_id IN ({$list})", $this->item_type );

		if ( $this->hide_for_reporters_enabled() ) {
			$where_conditions[] = "q.item_id IN ({$this->get_sql_for_reported_items( get_current_user_id(), $this->item_type )})";
		}

		if ( $this->hide_for_all_enabled() ) {
			$where_conditions[] = "q.item_id IN ({$this->get_sql_for_hidden_items( $this->item_type )})";
		}

		if ( ! empty( $where_conditions ) ) {
			$comment_ids_sql .= ' AND (' . join( ' OR ', $where_conditions ) . ')';
		}

		return $wpdb->get_col( $comment_ids_sql );
	}

	/**
	 * Cache hidden items from the given list of items.
	 *
	 * @param array $item_ids item ids.
	 */
	protected function cache_hidden_items( $item_ids ) {
		$hidden_ids = $this->get_hidden_ids( $item_ids );
		$this->_cache_ids( $hidden_ids );
	}

	/**
	 * Cache the given ids as hidden.
	 *
	 * @param array $ids ids.
	 */
	protected function _cache_ids( $ids ) {
		$cache = bpmts_moderation_tools()->cache;
		foreach ( $ids as $id ) {
			$cache->set( $this->item_type . '_' . $id, 1 );
		}
	}
}
