<?php
/**
 * Short Description
 *
 * @package    bp29_dev
 * @subpackage ${NAMESPACE}
 * @copyright  Copyright (c) 2018, Brajesh Singh
 * @license    https://www.gnu.org/licenses/gpl.html GNU Public License
 * @author     Brajesh Singh
 * @since      1.0.0
 */

namespace PressThemes\ModerationTools\Messages;

use PressThemes\ModerationTools\Models\Report;
use PressThemes\ModerationTools\Reportable_Types\Reportable_Item_Type;

/**
 * Message Base class.
 */
abstract class Message_Builder {

	/**
	 * Message subject template.
	 *
	 * @var string
	 */
	protected $subject = '';

	/**
	 * Message content template
	 *
	 * @var string
	 */
	protected $message = '';

	/**
	 * Supported notification channels.
	 *
	 * @var array
	 */
	protected $channels = array( 'email' );

	/**
	 * All message Tokens.
	 *
	 * @var array
	 */
	private $tokens = array();

	/**
	 * Message constructor.
	 *
	 * @param string $subject subject.
	 * @param string $message message.
	 */
	public function __construct( $subject, $message ) {
		$this->subject = $subject;
		$this->message = $message;
	}

	/**
	 * Parses subject. content and replaces token.
	 *
	 * @param \WP_User             $user user object.
	 * @param Reportable_Item_Type $item  item.
	 *
	 * @return Message
	 */
	public function parse( $user, $item ) {
		$user_tokens = $this->user_tokens_map( $user );
		$user_keys = array_keys( $user_tokens );
		$user_replacements = array_values( $user_tokens );

		$subject = str_replace( $user_keys, $user_replacements, $this->subject );
		$message = str_replace( $user_keys, $user_replacements, $this->message );

		$item_tokens = $this->all_item_tokens_map( $item );

		$item_keys = array_keys( $item_tokens );
		$item_replacements = array_values( $item_tokens );

		$subject = str_replace( $item_keys, $item_replacements, $subject );
		$message = str_replace( $item_keys, $item_replacements, $message );

		return new Message( $subject, $message );
	}

	/**
	 * Get the available tokens map for user.
	 *
	 * @param \WP_User $user user object.
	 *
	 * @return array
	 */
	protected function user_tokens_map( $user ) {
		return array(
			'{user.username}'    => $user->user_login,
			'{user.firstname}'   => $user->first_name,
			'{user.lastname}'    => $user->last_name,
			'{user.displayname}' => $user->display_name,
			'{user.url}'         => '',
		);
	}

	/**
	 * Retrieve all token.
	 *
	 * @param mixed $item item.
	 *
	 * @return array
	 */
	private function all_item_tokens_map( $item ) {
		$this->add_token_map( $this->item_tokens_map( $item ) );
		return $this->tokens;
	}

	/**
	 * Add token parser for report.
	 *
	 * @param Report |int $report report id or object.
	 */
	public function add_reports_token( $report ) {

		if ( is_int( $report ) ) {
			$report = Report::find( $report );
		}

		if ( ! $report ) {
			return;
		}

		$report_tokens = array(
			'{report.date}'     => mysql2date( 'F j, Y', $report->created_at ),
			'{report.content}'  => esc_html( $report->subject ) . '\n\n' . esc_html( $report->message ),
			'{report.link}'     => bpmts_admin_get_queue_entry_url( array( 'queue_id' => $report->queue_id ) ),
			'{report.reporter}' => $report->reporter_id ? bp_core_get_user_domain( $report->reporter_id ) : $report->name,
		);

		$this->add_token_map( $report_tokens );
	}

	/**
	 * Add tokens map.
	 *
	 * @param array $map token maps.
	 */
	public function add_token_map( $map ) {
		$this->tokens = array_merge( $this->tokens, $map );
	}

	/**
	 * Item tokens parser.
	 *
	 * @param  Reportable_Item_Type $item reported item.
	 *
	 * @return array
	 */
	abstract protected function item_tokens_map( $item );
}

