<?php
/**
 * BuddyPress Moderation Tools:- Topic Action handler.
 *
 * @package    BuddyPress Moderation Tools
 * @subpackage BBPress\Topic
 * @copyright  Copyright (c) 2018, Brajesh Singh
 * @license    https://www.gnu.org/licenses/gpl.html GNU Public License
 * @author     Brajesh Singh, Ravi Sharma
 * @since      1.0.0
 */

namespace PressThemes\ModerationTools\Reportable_Types\BBPress\Topic\Handlers;

use PressThemes\ModerationTools\Messages\Message_Builder;
use PressThemes\ModerationTools\Models\Queue;
use PressThemes\ModerationTools\Models\Report;
use PressThemes\ModerationTools\Reportable_Types\BBPress\Topic\Messages\Topic_Message_Factory;

// Do not allow direct access over web.
defined( 'ABSPATH' ) || exit;

/**
 * Action handler.
 */
class Topic_Action_Handler {

	/**
	 * Item type.
	 *
	 * @var string
	 */
	private $item_type = '';

	/**
	 * Boot this class.
	 */
	public static function boot() {
		$self            = new self();
		$self->item_type = bbp_get_topic_post_type();
		$self->setup();
	}

	/**
	 * Setup hooks callbacks
	 */
	public function setup() {
		add_action( "bpmts_{$this->item_type}_reported", array( $this, 'on_report' ), 10, 3 );
		add_action( "bpmts_{$this->item_type}_restored", array( $this, 'on_restore' ) );
		add_action( "bpmts_{$this->item_type}_hidden", array( $this, 'on_hide' ), 10, 2 );
		add_action( "bpmts_{$this->item_type}_deleted", array( $this, 'on_delete' ), 10, 2 );

		// for access protection.
		if ( ! is_super_admin() ) {
			add_action( 'template_redirect', array( $this, 'check_access' ) );
		}
	}

	/**
	 * Disable access to hidden forums? for everyone?
	 */
	public function check_access() {
		// @todo recheck.
		if ( ! is_singular( $this->item_type ) || ! bpmts_is_item_hidden( get_queried_object_id(), $this->item_type ) ) {
			return;
		}

		$post_type_object = get_post_type_object( $this->item_type );
		$label            = $post_type_object->labels->singular_name;

		/* translators: %s: topic title */
		bp_core_add_message( sprintf( __( 'The %s is unavailable currently. Please try again later.', 'buddypress-moderation-tools' ), $label ), 'error' );

		$redirect_location = is_user_logged_in() ? bp_loggedin_user_domain() : home_url( '/' );
		// @todo add a filter here.
		bp_core_redirect( $redirect_location );
	}

	/**
	 * Notify admins via email on new report
	 *
	 * @param int    $topic_id id of topic whose status has changed.
	 * @param Queue  $queue queue entry.
	 * @param Report $report report object.
	 */
	public function on_report( $topic_id, $queue, $report ) {

		// on Restore, send email to user.
		if ( bpmts_get_option( 'bbp_topic_notify_admin_on_report' ) ) {
			$message_builder = Topic_Message_Factory::build( 'admin-reported' );

			if ( ! $message_builder ) {
				return;
			}

			$message_builder->add_reports_token( $report );
			$this->notify_admin( $topic_id, $message_builder );
		}
	}

	/**
	 * Notify user via email on account being hidden.
	 *
	 * @param int   $topic_id id of forum whose status has changed.
	 * @param Queue $queue queue entry.
	 */
	public function on_hide( $topic_id, $queue ) {

		$is_super_admin = is_super_admin();

		// on Hide, sed email if allowed.
		if ( bpmts_get_option( 'bbp_topic_notify_author_on_hide' ) ) {
			$message_builder = Topic_Message_Factory::build( 'user-hidden' );

			if ( ! $message_builder ) {
				return;
			}

			$this->notify_user( $topic_id, $message_builder );
		}

		// on Hide.
		if ( ! $is_super_admin && bpmts_get_option( 'bbp_topic_notify_admin_on_hide' ) ) {
			$message_builder = Topic_Message_Factory::build( 'admin-hidden' );

			if ( ! $message_builder ) {
				return;
			}

			$message_builder->add_token_map( array( '{queue.entry}' => bpmts_admin_get_queue_entry_url( array( 'queue_id' => $queue->id ) ) ) );

			$this->notify_admin( $topic_id, $message_builder );
		}
	}

	/**
	 * Notify user via email on account activation
	 *
	 * @param int $topic_id id of topic whose status has changed.
	 */
	public function on_restore( $topic_id ) {

		// Restored, send email to user.
		if ( bpmts_get_option( 'bbp_topic_notify_author_on_restore' ) ) {
			$message_builder = Topic_Message_Factory::build( 'user-restored' );

			if ( ! $message_builder ) {
				return;
			}

			$this->notify_user( $topic_id, $message_builder );
		}
	}

	/**
	 * Notify user via email on topic deletion.
	 *
	 * @param int      $topic_id id of topic whose status has changed.
	 * @param \WP_Post $topic Topic object.
	 */
	public function on_delete( $topic_id, $topic ) {

		// Restored, send email to user.
		if ( bpmts_get_option( 'bbp_topic_notify_author_on_delete' ) ) {
			$message_builder = Topic_Message_Factory::build( 'user-deleted' );

			if ( ! $message_builder ) {
				return;
			}
			$this->notify_user( $topic, $message_builder );
		}
	}

	/**
	 * Notify admin on new report or hidden.
	 *
	 * @param int             $topic_id id of forum whose status has changed.
	 * @param Message_Builder $message_builder message builder.
	 */
	public function notify_admin( $topic_id, $message_builder ) {
		$topic = bbp_get_topic( $topic_id );

		$administrators = get_users(
			array(
				'role' => 'administrator',
			)
		);

		foreach ( $administrators as $administrator ) {
			$message = $message_builder->parse( $administrator, $topic );

			if ( ! $message ) {
				continue;
			}

			wp_mail( $administrator->user_email, $message->subject, $message->message );
		}
	}

	/**
	 * Notify author via email on post type hidden/restore.
	 *
	 * @param int             $topic_id id of Topic whose status has changed.
	 * @param Message_Builder $message_builder message builder.
	 */
	public function notify_user( $topic_id, $message_builder ) {
		$topic = bbp_get_topic( $topic_id );

		$user_id = $topic->post_author;
		$user    = get_userdata( $user_id );

		if ( ! $user ) {
			return;
		}

		$message = $message_builder->parse( $user, $topic );
		wp_mail( $user->user_email, $message->subject, $message->message );
	}
}
