<?php
/**
 * ReportAbuse:- User Action handler.
 *
 * @package    ReportAbuse
 * @subpackage User
 * @copyright  Copyright (c) 2018, Brajesh Singh
 * @license    https://www.gnu.org/licenses/gpl.html GNU Public License
 * @author     Brajesh Singh
 * @since      1.0.0
 */

namespace PressThemes\ModerationTools\Reportable_Types\BuddyPress\Message\Handlers;

// Exit if files access directly over web.
use PressThemes\ModerationTools\Models\Queue;
use PressThemes\ModerationTools\Reportable_Types\BuddyPress\Group\Messages\Group_Message_Builder;
use PressThemes\ModerationTools\Reportable_Types\BuddyPress\Group\Messages\Group_Message_Factory;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Action handler.
 */
class Message_Action_Handler {

	/**
	 * Boot this class.
	 */
	public static function boot() {
		$self = new self();
		$self->setup();
	}

	/**
	 * Setup hooks callbacks
	 */
	public function setup() {

		add_action( 'bpmts_bp_group_reported', array( $this, 'on_report' ), 10, 3 );
		add_action( 'bpmts_bp_group_restored', array( $this, 'on_restore' ) );
		add_action( 'bpmts_bp_group_hidden', array( $this, 'on_hide' ), 10, 2 );
	}

	/**
	 * Notify admins via email on new report
	 *
	 * @param int $group_id id of group whose status has changed.
	 */
	public function on_report( $group_id, $queue, $report ) {

		// Account activated, send email to user.
		if ( bpmts_get_option( 'bp_group_notify_admin_on_report' ) ) {
			$message_builder = Group_Message_Factory::build( 'admin-reported' );

			if ( ! $message_builder ) {
				return;
			}

			$message_builder->add_reports_token( $report );
			$this->notify_admin( $group_id, $message_builder );
		}
	}

	/**
	 * Notify user via email on account being hidden.
	 *
	 * @param int   $group_id id of group whose status has changed.
	 * @param Queue $queue queue entry.
	 */
	public function on_hide( $group_id, $queue ) {

		$is_super_admin = is_super_admin();

		// Account deactivated, sed email if allowed.
		if ( bpmts_get_option( 'bp_group_notify_user_on_hide' ) ) {
			$message_builder = Group_Message_Factory::build( 'user-hidden' );

			if ( ! $message_builder ) {
				return;
			}
			$message_builder->add_token_map( array( '{queue.entry}' => bpmts_admin_get_queue_entry_url( array( 'queue_id' => $queue->id ) ) ) );
			$this->notify_user( $group_id, $message_builder );
		}

		// Account deactivated by user.
		if ( ! $is_super_admin && bpmts_get_option( 'bp_group_notify_admin_on_hide' ) ) {
			$message_builder = Group_Message_Factory::build( 'admin-hidden' );

			if ( ! $message_builder ) {
				return;
			}

			$this->notify_admin( $group_id, $message_builder );
		}
	}

	/**
	 * Notify user via email on account activation
	 *
	 * @param int $group_id id of group whose status has changed.
	 */
	public function on_restore( $group_id ) {

		// Account activated, send email to user.
		if ( bpmts_get_option( 'bp_group_notify_user_on_restore' ) ) {
			$message_builder = Group_Message_Factory::build( 'user-restored' );

			if ( ! $message_builder ) {
				return;
			}
			$this->notify_user( $group_id, $message_builder );
		}
	}

	/**
	 * Notify admin on new report or hidden.
	 *
	 * @param int                   $group_id id of group whose status has changed.
	 * @param Group_Message_Builder $message_builder account_activation or account_deactivate.
	 */
	public function notify_admin( $group_id, $message_builder ) {

		$group          = new \BP_Groups_Group( $group_id );
		$administrators = get_users( array(
			'role' => 'administrator',
		) );


		foreach ( $administrators as $administrator ) {
			$message = $message_builder->parse( $administrator, $group );
			if ( ! $message ) {
				continue;
			}

			wp_mail( $administrator->user_email, $message->subject, $message->message );
		}
	}

	/**
	 * Notify all group admin users via email on group hidden/restore.
	 *
	 * @param int                   $group_id id of group whose status has changed.
	 * @param Group_Message_Builder $message_builder account_activation or account_deactivate.
	 */
	public function notify_user( $group_id, $message_builder ) {

		$group = new \BP_Groups_Group( $group_id );

		$user_ids = wp_list_pluck( $group->admins, 'user_id' );
		foreach ( $user_ids as $user_id ) {
			$user = get_userdata( $user_id );
			if ( ! $user ) {
				continue;
			}
			$message = $message_builder->parse( $user, $group );
			wp_mail( $user->user_email, $message->subject, $message->message );
		}
	}

}
