<?php
/**
 * ReportAbuse:- Comment Type Action handler.
 *
 * @package    ModerationTools
 * @subpackage Post_Type
 * @copyright  Copyright (c) 2018, Brajesh Singh
 * @license    https://www.gnu.org/licenses/gpl.html GNU Public License
 * @author     Brajesh Singh
 * @since      1.0.0
 */

namespace PressThemes\ModerationTools\Reportable_Types\Comment_Type\Handlers;

use PressThemes\ModerationTools\Messages\Message_Builder;
use PressThemes\ModerationTools\Models\Queue;
use PressThemes\ModerationTools\Reportable_Types\Comment_Type\Messages\Comment_Type_Message_Factory;

// Exit if files access directly over web.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Action handler.
 */
class Comment_Type_Action_Handler {

	/**
	 * Post type.
	 *
	 * @var string
	 */
	private $comment_type = '';

	/**
	 * Comment label
	 *
	 * @var string
	 */
	private $comment_label = '';

	/**
	 * Boot this class.
	 *
	 * @param string $comment_type comment type.
	 * @param string $label Label.
	 */
	public static function boot( $comment_type, $label ) {
		$self                = new self();
		$self->comment_type  = $comment_type;
		$self->comment_label = $label;

		$self->setup();
	}

	/**
	 * Setup hooks callbacks
	 */
	public function setup() {

		add_action( "bpmts_{$this->comment_type}_reported", array( $this, 'on_report' ), 10, 3 );
		add_action( "bpmts_{$this->comment_type}_restored", array( $this, 'on_restore' ) );
		add_action( "bpmts_{$this->comment_type}_hidden", array( $this, 'on_hide' ), 10, 2 );

		// for access protection.
		if ( ! is_super_admin() ) {
			add_action( 'template_redirect', array( $this, 'check_access' ) );
		}
	}

	/**
	 * Disable access to hidden comments? for everyone?
	 */
	public function check_access() {

		if ( ! bpmts_is_item_hidden( get_queried_object_id(), $this->comment_type ) ) {
			return;
		}

		$label            = $this->comment_label;
		bp_core_add_message( sprintf( __( 'The %s is unavailable currently. Please try again later', 'buddypress-moderation-tolls' ), $label ), 'error' );

		$redirect_location = is_user_logged_in() ? bp_loggedin_user_domain() : home_url( '/' );
		// @todo add a filter here.
		bp_core_redirect( $redirect_location );
	}

	/**
	 * Notify admins via email on new report
	 *
	 * @param int $comment_id id of group whose status has changed.
	 */
	public function on_report( $comment_id, $queue, $report ) {

		// on Restore, send email to user.
		if ( bpmts_get_option( "{$this->comment_type}_notify_admin_on_report" ) ) {
			$message_builder = Comment_Type_Message_Factory::build( 'admin-reported', $this->comment_type );

			if ( ! $message_builder ) {
				return;
			}

			$message_builder->add_reports_token( $report );
			$this->notify_admin( $comment_id, $message_builder );
		}
	}

	/**
	 * Notify user via email on account being hidden.
	 *
	 * @param int   $comment_id id of post whose status has changed.
	 * @param Queue $queue queue entry.
	 */
	public function on_hide( $comment_id, $queue ) {

		$is_super_admin = is_super_admin();

		// on Hide, sed email if allowed.
		if ( bpmts_get_option( "{$this->comment_type}_notify_author_on_hide" ) ) {
			$message_builder = Comment_Type_Message_Factory::build( 'user-hidden', $this->comment_type );

			if ( ! $message_builder ) {
				return;
			}

			$this->notify_user( $comment_id, $message_builder );
		}

		// on Hide.
		if ( ! $is_super_admin && bpmts_get_option( "{$this->comment_type}_notify_admin_on_hide" ) ) {
			$message_builder = Comment_Type_Message_Factory::build( 'admin-hidden', $this->comment_type );

			if ( ! $message_builder ) {
				return;
			}

			$message_builder->add_token_map( array( '{queue.entry}' => bpmts_admin_get_queue_entry_url( array( 'queue_id' => $queue->id ) ) ) );

			$this->notify_admin( $comment_id, $message_builder );
		}
	}

	/**
	 * Notify user via email on account activation
	 *
	 * @param int $comment_id id of comment whose status has changed.
	 */
	public function on_restore( $comment_id ) {

		// Restored, send email to user.
		if ( bpmts_get_option( "{$this->comment_type}_notify_author_on_restore" ) ) {
			$message_builder = Comment_Type_Message_Factory::build( 'user-restored', $this->comment_type );

			if ( ! $message_builder ) {
				return;
			}
			$this->notify_user( $comment_id, $message_builder );
		}
	}

	/**
	 * Notify admin on new report or hidden.
	 *
	 * @param int             $comment_id id of comment whose status has changed.
	 * @param Message_Builder $message_builder message builder.
	 */
	public function notify_admin( $comment_id, $message_builder ) {

		$comment        = get_comment( $comment_id );
		$administrators = get_users( array(
			'role' => 'administrator',
		) );

		foreach ( $administrators as $administrator ) {
			$message = $message_builder->parse( $administrator, $comment );
			if ( ! $message ) {
				continue;
			}

			wp_mail( $administrator->user_email, $message->subject, $message->message );
		}
	}

	/**
	 * Notify author via email on comment type hidden/restore.
	 *
	 * @param int             $comment_id id of comment whose status has changed.
	 * @param Message_Builder $message_builder message builder.
	 */
	public function notify_user( $comment_id, $message_builder ) {

		$comment = get_comment( $comment_id );

		$user_id = $comment->user_id;
		$user    = get_userdata( $user_id );
		if ( ! $user ) {
			return;
		}

		$message = $message_builder->parse( $user, $comment );
		wp_mail( $user->user_email, $message->subject, $message->message );
	}
}
