<?php
/**
 * Reportable support for post types.
 *
 * @package    ReportAbuse
 * @subpackage Reportable
 * @copyright  Copyright (c) 2018, Brajesh Singh
 * @license    https://www.gnu.org/licenses/gpl.html GNU Public License
 * @author     Brajesh Singh
 * @since      1.0.0
 */

namespace PressThemes\ModerationTools\Reportable_Types\Post_Type;

use PressThemes\ModerationTools\Reportable_Types\Post_Type\Filters\Post_Type_Query_Filter;
use PressThemes\ModerationTools\Reportable_Types\Post_Type\Handlers\Post_Type_Action_Handler;
use PressThemes\ModerationTools\Reportable_Types\Post_Type\Handlers\Post_Type_UI_Helper;
use PressThemes\ModerationTools\Reportable_Types\Reportable_Item_Type;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Reportable post type
 */
class Reportable_Post_Type extends Reportable_Item_Type {

	/**
	 * Reportable_Post_Type constructor.
	 *
	 * @param string $post_type post type.
	 */
	public function __construct( $post_type ) {
		$post_type_obj = get_post_type_object( $post_type );
		parent::__construct( $post_type, $post_type_obj->labels->name );
	}


	/**
	 * Boot this module.
	 *
	 * Load the dependencies needed.
	 */
	public function boot() {
		$path = dirname( __FILE__ );
		//require_once $path . '/group/bpmts-group-functions.php';
	}

	/**
	 * Setup hooks. Called after boot.
	 */
	public function setup() {
		// do some setup.
		// Boot action handlers.
		Post_Type_Action_Handler::boot( $this->type );
		Post_Type_Query_Filter::boot( $this->type );
		// Boot UI Helper to add messages/buttons as needed.
		Post_Type_UI_Helper::boot( $this->type );
	}

	/**
	 * Add admin settings.
	 *
	 * @param \Press_Themes\PT_Settings\Page $page page object.
	 */
	public function add_admin_settings( $page ) {
		$panel = $page->add_panel( 'post-type-' . $this->type, $this->label );
		Post_Type_Settings_Panel::build( $panel, $this->type );
	}


	/**
	 * Check that post type is registered and is public.
	 *
	 * @param int    $item_id post id.
	 * @param string $context ignore.
	 * @param int    $context_id ignore.
	 *
	 * @return bool
	 */
	public function is_valid( $item_id, $context, $context_id ) {
		$post_type_obj = get_post_type_object( $this->type );
		return $post_type_obj && $post_type_obj->public;
	}
	/**
	 * Get user threshold.
	 *
	 * @return int
	 */
	public function get_threshold() {
		return bpmts_get_option( 'post_type_' . $this->type . '_threshold', 5 );
	}

	/**
	 * Delete an item of this type.
	 *
	 * @param int $item_id item id.
	 */
	public function delete( $item_id ) {
		wp_delete_post( $item_id, true );
	}

	/**
	 * Get the item object.
	 *
	 * @param int $item_id user id.
	 *
	 * @return false|\WP_Post
	 */
	public function get_item( $item_id ) {
		return get_post( $item_id );
	}

	/**
	 * Get the thumbnail
	 *
	 * @param int    $item_id item id.
	 * @param string $size thumb|full.
	 *
	 * @return string
	 */
	public function get_item_thumbnail( $item_id, $size ) {
		return get_the_post_thumbnail( $item_id, $size );
	}

	/**
	 * Get item title.
	 *
	 * @param int $item_id item id.
	 *
	 * @return string
	 */
	public function get_item_title( $item_id ) {
		return get_the_title( $item_id );
	}

	/**
	 * Get post description.
	 *
	 * @param int $item_id item id.
	 *
	 * @return string
	 */
	public function get_item_description( $item_id ) {
		$post = get_post( $item_id );
		return $post->post_excerpt ? get_the_excerpt( $post ) : wp_trim_excerpt( $post->post_content );
	}

	/**
	 * Get Post Permalink.
	 *
	 * @param int $item_id user id.
	 *
	 * @return false|string
	 */
	public function get_item_permalink( $item_id ) {
		return get_permalink( $item_id );
	}

	/**
	 * Get item edit url.
	 *
	 * @param int $item_id item id.
	 *
	 * @return string
	 */
	public function get_item_edit_url( $item_id ) {
		return get_edit_post_link( $item_id );
	}

	/**
	 * Get item delete url.
	 *
	 * @param int $item_id item id.
	 *
	 * @return string
	 */
	public function get_item_delete_url( $item_id ) {
		return get_delete_post_link( $item_id );
	}
}
