<?php
/**
 * Post Type filter.
 *
 * @package    BuddyPress Moderation Tools
 * @subpackage Post_Type
 * @copyright  Copyright (c) 2018, Brajesh Singh
 * @license    https://www.gnu.org/licenses/gpl.html GNU Public License
 * @author     Brajesh Singh
 * @since      1.0.0
 */

namespace PressThemes\ModerationTools\Reportable_Types\Post_Type\Filters;

use PressThemes\ModerationTools\Filters\Query_Filter;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Filter Post type.
 */
class Post_Type_Query_Filter extends Query_Filter {

	/**
	 * Post type.
	 *
	 * @var string
	 */
	private $post_type;

	/**
	 * Boot this class.
	 *
	 * @param string $post_type post type.
	 */
	public static function boot( $post_type ) {
		$self            = new self();
		$self->post_type = $post_type;
		$self->setup();
	}

	/**
	 * Setup hooks callbacks
	 */
	public function setup() {
		$this->settings = array(
			'auto_hide' => bpmts_get_option( "post_type_{$this->post_type}_auto_hide", 0 ),
			'hide_type' => bpmts_get_option( "post_type_{$this->post_type}_hide_type", 'all' ),
		);

		// hide posts from list everywhere.
		add_action( 'posts_where', array( $this, 'exclude_in_listing' ), 200, 2 );
	}

	/**
	 * Exclude posts from query.
	 *
	 * @param string    $sql Concatenated SQL statement.
	 * @param \WP_Query $query Query object.
	 *
	 * @return string
	 */
	public function exclude_in_listing( $sql, $query ) {
		// do not hide users inside the admin.
		if ( is_super_admin() || ( is_admin() && ! defined( 'DOING_AJAX' ) ) ) {
			return $sql;
		}

		$queried_post_type = $query->get( 'post_type' );
		$hide              = true;

		if ( empty( $queried_post_type ) ) {
			$hide = true;
		} elseif ( is_array( $queried_post_type ) && ! in_array( $this->post_type, $queried_post_type, true ) ) {
			$hide = false;
		} elseif ( $queried_post_type !== $this->post_type ) {
			$hide = false;
		}

		if ( ! $hide ) {
			return $sql;
		}

		// if filter is not needed.
		if ( $this->hide_for_reporters_enabled() ) {
			$hidden_reporters_sql = $this->get_sql_for_reported_items( get_current_user_id(), $this->post_type );

			$sql .= " AND ID NOT IN($hidden_reporters_sql)";
		}

		if ( $this->hide_for_all_enabled() ) {
			$hidden_all_sql = $this->get_sql_for_hidden_items( $this->post_type );

			$sql .= " AND ID NOT IN ({$hidden_all_sql}) ";
		}


		return $sql;
	}
}
