<?php
/**
 * ReportAbuse:- Post Type Action handler.
 *
 * @package    ReportAbuse
 * @subpackage Post_Type
 * @copyright  Copyright (c) 2018, Brajesh Singh
 * @license    https://www.gnu.org/licenses/gpl.html GNU Public License
 * @author     Brajesh Singh
 * @since      1.0.0
 */

namespace PressThemes\ModerationTools\Reportable_Types\Post_Type\Handlers;

use PressThemes\ModerationTools\Messages\Message_Builder;
use PressThemes\ModerationTools\Models\Queue;
use PressThemes\ModerationTools\Models\Report;
use PressThemes\ModerationTools\Reportable_Types\Post_Type\Messages\Post_Type_Message_Factory;

// Do not allow direct access over web.
defined( 'ABSPATH' ) || exit;

/**
 * Action handler.
 */
class Post_Type_Action_Handler {

	/**
	 * Post type.
	 *
	 * @var string
	 */
	private $post_type = '';

	/**
	 * Boot this class.
	 *
	 * @param string $post_type post type.
	 */
	public static function boot( $post_type ) {
		$self            = new self();
		$self->post_type = $post_type;
		$self->setup();
	}

	/**
	 * Setup hooks callbacks
	 */
	public function setup() {

		add_action( "bpmts_{$this->post_type}_reported", array( $this, 'on_report' ), 10, 3 );
		add_action( "bpmts_{$this->post_type}_restored", array( $this, 'on_restore' ) );
		add_action( "bpmts_{$this->post_type}_hidden", array( $this, 'on_hide' ), 10, 2 );
		add_action( "bpmts_{$this->post_type}_deleted", array( $this, 'on_delete' ), 10, 2 );

		// for access protection.
		if ( ! is_super_admin() ) {
			add_action( 'template_redirect', array( $this, 'check_access' ) );
		}
	}

	/**
	 * Disable access to hidden posts? for everyone?
	 */
	public function check_access() {

		if ( ! is_singular( $this->post_type ) || ! bpmts_is_item_hidden( get_queried_object_id(), $this->post_type ) ) {
			return;
		}

		$post_type_object = get_post_type_object( $this->post_type );
		$label            = $post_type_object->labels->singular_name;
		bp_core_add_message( sprintf( __( 'The %s is unavailable currently. Please try again later', 'buddypress-moderation-tools' ), $label ), 'error' );

		$redirect_location = is_user_logged_in() ? bp_loggedin_user_domain() : home_url( '/' );
		// @todo add a filter here.
		bp_core_redirect( $redirect_location );
	}

	/**
	 * Notify admins via email on new report
	 *
	 * @param int    $post_id id of group whose status has changed.
	 * @param Queue  $queue queue entry.
	 * @param Report $report report object.
	 */
	public function on_report( $post_id, $queue, $report ) {

		// on Restore, send email to user.
		if ( bpmts_get_option( "post_type_{$this->post_type}_notify_admin_on_report" ) ) {
			$message_builder = Post_Type_Message_Factory::build( 'admin-reported', $this->post_type );

			if ( ! $message_builder ) {
				return;
			}

			$message_builder->add_reports_token( $report );
			$this->notify_admin( $post_id, $message_builder );
		}
	}

	/**
	 * Notify user via email on account being hidden.
	 *
	 * @param int   $post_id id of post whose status has changed.
	 * @param Queue $queue queue entry.
	 */
	public function on_hide( $post_id, $queue ) {

		$is_super_admin = is_super_admin();

		// on Hide, sed email if allowed.
		if ( bpmts_get_option( "post_type_{$this->post_type}_notify_author_on_hide" ) ) {
			$message_builder = Post_Type_Message_Factory::build( 'user-hidden', $this->post_type );

			if ( ! $message_builder ) {
				return;
			}

			$this->notify_user( $post_id, $message_builder );
		}

		// on Hide.
		if ( ! $is_super_admin && bpmts_get_option( "post_type_{$this->post_type}_notify_admin_on_hide" ) ) {
			$message_builder = Post_Type_Message_Factory::build( 'admin-hidden', $this->post_type );

			if ( ! $message_builder ) {
				return;
			}

			$message_builder->add_token_map( array( '{queue.entry}' => bpmts_admin_get_queue_entry_url( array( 'queue_id' => $queue->id ) ) ) );

			$this->notify_admin( $post_id, $message_builder );
		}
	}

	/**
	 * Notify user via email on account activation
	 *
	 * @param int $post_id id of post whose status has changed.
	 */
	public function on_restore( $post_id ) {

		// Restored, send email to user.
		if ( bpmts_get_option( "post_type_{$this->post_type}_notify_author_on_restore" ) ) {
			$message_builder = Post_Type_Message_Factory::build( 'user-restored', $this->post_type );

			if ( ! $message_builder ) {
				return;
			}
			$this->notify_user( $post_id, $message_builder );
		}
	}

	/**
	 * Notify user via email on post deletion.
	 *
	 * @param int      $post_id id of post whose status has changed.
	 * @param \WP_Post $post Post object.
	 */
	public function on_delete( $post_id, $post ) {

		// Restored, send email to user.
		if ( bpmts_get_option( "post_type_{$this->post_type}_notify_author_on_delete" ) ) {
			$message_builder = Post_Type_Message_Factory::build( 'user-deleted', $this->post_type );

			if ( ! $message_builder ) {
				return;
			}
			$this->notify_user( $post, $message_builder );
		}
	}

	/**
	 * Notify admin on new report or hidden.
	 *
	 * @param int             $post_id id of post whose status has changed.
	 * @param Message_Builder $message_builder message builder.
	 */
	public function notify_admin( $post_id, $message_builder ) {

		$post          = get_post( $post_id );
		$administrators = get_users( array(
			'role' => 'administrator',
		) );

		foreach ( $administrators as $administrator ) {
			$message = $message_builder->parse( $administrator, $post );
			if ( ! $message ) {
				continue;
			}

			wp_mail( $administrator->user_email, $message->subject, $message->message );
		}
	}

	/**
	 * Notify author via email on post type hidden/restore.
	 *
	 * @param int             $post_id id of post whose status has changed.
	 * @param Message_Builder $message_builder message builder.
	 */
	public function notify_user( $post_id, $message_builder ) {

		$post = get_post( $post_id );

		$user_id = $post->post_author;
		$user    = get_userdata( $user_id );
		if ( ! $user ) {
			return;
		}

		$message = $message_builder->parse( $user, $post );
		wp_mail( $user->user_email, $message->subject, $message->message );
	}

}
