<?php
/**
 * Moderation Tools/Report Abuse Database Schema helper
 *
 * @package    ModerationTools
 * @subpackage Schema
 * @copyright  Copyright (c) 2018, Brajesh Singh
 * @license    https://www.gnu.org/licenses/gpl.html GNU Public License
 * @author     Brajesh Singh
 * @since      1.0.0
 */

namespace PressThemes\ModerationTools\Schema;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Moderation Tools Schema Manager.
 *
 * For actual models, Please see models directory.
 */
class Schema {

	/**
	 * Get table name.
	 *
	 * @param string $name table identifier.
	 *
	 * @return null|string full table name or null.
	 */
	public static function table( $name ) {
		$tables = array(
			'queue'    => 'bpmts_moderation_queue', // queue.
			'reports'  => 'bpmts_moderation_reports', // reports.
			'category' => 'bpmts_moderation_categories', // categories.
			'log'      => 'bpmts_moderation_log', // Moderation log.
		);

		global $wpdb;

		return isset( $tables[ $name ] ) ? $wpdb->prefix . $tables[ $name ] : null;
	}

	/**
	 * Create Tables.
	 */
	public static function create() {
		global $wpdb;

		require_once( ABSPATH . 'wp-admin/includes/upgrade.php' );

		$charset_collate = $wpdb->get_charset_collate();

		$table_queue      = self::table( 'queue' );
		$table_reports    = self::table( 'reports' );
		$table_log        = self::table( 'log' );
		$table_categories = self::table( 'category' );

		$sql = array();

		if ( ! self::exists( $table_queue ) ) {
			$sql[] = "CREATE TABLE `{$table_queue}`(
  				`id` bigint(20) NOT NULL AUTO_INCREMENT PRIMARY KEY,
  				`user_id` bigint(20) NOT NULL,
  				`item_id` bigint(20) NOT NULL,
  				`item_type` varchar(32) NOT NULL,
  				`context_id` bigint(20) DEFAULT '0',
  				`context` varchar(32) DEFAULT '',
  				`is_new` tinyint(1) NOT NULL DEFAULT '1',
  				`is_hidden` tinyint(1) NOT NULL DEFAULT '0',
  				`total_reports_count` mediumint(11) NOT NULL DEFAULT '0',
  				`current_reports_count` int(11) NOT NULL DEFAULT '0',
  				`created_at` datetime NOT NULL,
  				`updated_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
  				 KEY `item_id_type` (`item_id`,`item_type`),
  				 KEY `context` (`context`,`context_id`)
			){$charset_collate};";
		}

		if ( ! self::exists( $table_reports ) ) {
			$sql[] = "CREATE TABLE `{$table_reports}` (
  				`id` bigint(20) NOT NULL AUTO_INCREMENT PRIMARY KEY,
  				`queue_id` bigint(20) NOT NULL,
  				`log_id` bigint(20) NOT NULL DEFAULT '0',
  				`reporter_id` bigint(20) DEFAULT NULL,
  				`is_auto` tinyint(1) NOT NULL DEFAULT '0',
  				`name` varchar(50) DEFAULT NULL,
  				`email` varchar(100) DEFAULT NULL,
  				`ip` varchar(20) DEFAULT NULL,
  				`subject` varchar(100) DEFAULT NULL,
  				`message` text,
  				`category` smallint(6) DEFAULT NULL,
  				`created_at` datetime NOT NULL,
  				`updated_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP
			){$charset_collate};";
		}

		if ( ! self::exists( $table_log ) ) {

			$sql[] = "CREATE TABLE `{$table_log}` (
  				`id` bigint(20) NOT NULL AUTO_INCREMENT PRIMARY KEY,
  				`queue_id` bigint(20) NOT NULL,
  				`moderator_id` bigint(20) NOT NULL,
  				`action` varchar(20) DEFAULT NULL,
  				`message` text NOT NULL,
  				`created_at` datetime NOT NULL,
  				`updated_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
  				 KEY `moderator_id` (`moderator_id`)
			){$charset_collate};";
		}

		if ( ! self::exists( $table_categories ) ) {
			$sql[] = "CREATE TABLE `{$table_categories}` (
  			`id` smallint(6) NOT NULL AUTO_INCREMENT PRIMARY KEY,
  			`name` varchar(128) NOT NULL,
  			`is_active` tinyint(4) NOT NULL DEFAULT '1',
  			`position` smallint(6) NOT NULL DEFAULT '0') {$charset_collate};";
		}

		if ( ! $sql ) {
			return;
		}

		dbDelta( $sql );

	}

	/**
	 * Check if table exists.
	 *
	 * @param string $table_name table name.
	 *
	 * @return bool
	 */
	public static function exists( $table_name ) {
		global $wpdb;

		return $wpdb->get_var( $wpdb->prepare( "SHOW TABLES LIKE %s", $table_name ) ) === $table_name;
	}
}
