<?php
if(!class_exists('EOD_API'))
{
    class EOD_API{
        public function __construct(){

        }

        /**
         * Get User API key (token)
         * @return string
         */
        public static function get_eod_api_key()
        {
            $plugin_options = get_option('eod_options');
            if($plugin_options === false) $plugin_options = array();
            //Default token
            $apiKey = EOD_DEFAULT_API;
            if(array_key_exists('api_key', $plugin_options) && $plugin_options['api_key']){
                $apiKey = $plugin_options['api_key'];
            }
            return($apiKey);
        }

        /**
         * Check the API key (token) and its tariff plan for the possibility of receiving data
         * @param string type
         * @param array props
         * @return mixed
         */
        public static function check_token_capability($type, $props)
        {
            if(($type === 'historical' || $type === 'live') && isset($props['target'])){
                return self::get_real_time_ticker($type, $props['target']);
            }

            if($type === 'news' && ($props['target'] || $props['tag'])){
                return self::get_news($props['target'], array(
                    'limit' => 1,
                    'tag'   => ''
                ));
            }

            if($type === 'fundamental' && $props['target']){
                return self::get_fundamental_data($props['target']);
            }

            return array();
        }


        /**
         * Check the API key (token) and its tariff plan for the possibility of receiving data
         * @param string target
         * @param array args
         * @return mixed
         */
        public static function get_news( $target, $args = array() ){
            // Check target/tag
            if((!isset($target) || gettype($target) !== 'string') && (!isset($args['tag']) || gettype($args['tag']) !== 'string'))
                return array('error' => 'wrong target');

            $apiKey = self::get_eod_api_key();

            // Base URL
            $apiUrl = "https://eodhistoricaldata.com/api/news?api_token=$apiKey";

            // Target
            if($target && gettype($target) === 'string')
                $apiUrl .= "&s=$target";
            // Tag
            if($args['tag'] && gettype($args['tag']) === 'string')
                $apiUrl .= "&t=".$args['tag'];

            // Offset
            $offset = isset($args['offset']) ? intval($args['offset']) : 0;
            if($offset < 0) $offset = 0;
            $apiUrl .= "&offset=$offset";

            // Limit
            $limit = isset($args['limit']) ? intval($args['limit']) : 50;
            if($limit < 1) $limit = 1;
            if($limit > 1000) $limit = 1000;
            $apiUrl .= "&limit=$limit";

            // Date range
            if($args['from']){
                $d = DateTime::createFromFormat('Y-m-d', $args['from']);
                if($d && $d->format('Y-m-d') === $args['from'])
                    $apiUrl .= "&from=".$args['from'];
            }
            if($args['to']){
                $d = DateTime::createFromFormat('Y-m-d', $args['to']);
                if($d && $d->format('Y-m-d') === $args['to'])
                    $apiUrl .= "&to=".$args['to'];
            }

            return self::call_eod_api($apiUrl);
        }


        /**
         * Searching for items from API by string
         * @param string $needle
         * @return mixed
         */
        public static function search_by_string($needle)
        {
            if(!$needle){
                return array('error' => 'empty string');
            }
            $needle = sanitize_text_field($needle);

//            $apiKey = self::get_eod_api_key();
//            $apiUrl = "https://eodhistoricaldata.com/api/search/".
//                $needle .
//                "?api_token=$apiKey";
            $apiUrl = "https://eodhistoricaldata.com/api/query-search-extended/?q=$needle";

            $data = self::call_eod_api($apiUrl);
            if(!$data)
                return array('error' => 'no result from api');

            return $data;
        }

        /**
         * Get Fundamental Data
         * @param string target
         * @return mixed
         */
        public static function get_fundamental_data($target)
        {
            if(!is_string($target)) return array('error' => 'Wrong target');

            $apiKey = self::get_eod_api_key();
            $apiUrl = "https://eodhistoricaldata.com/api/fundamentals/".
                strtoupper($target).
                "?api_token=$apiKey".
                "&fmt=json";

            $fundamental_data = self::call_eod_api($apiUrl);
            if(!$fundamental_data)
                return array('error' => 'no result from fundamental data api');

            return $fundamental_data;
        }


        /**
         * Get Ticker infos and calculate evolution
         * @param string type
         * @param mixed targets
         * @return mixed
         */
        public static function get_real_time_ticker($type = 'historical', $targets)
        {
            $apiKey = self::get_eod_api_key();

            if($type === 'historical'){
                if(!is_array($targets)){
                    $targets = array($targets);
                }
                $apiUrl = "https://eodhistoricaldata.com/api/eod-bulk-last-day/US".
                    "?api_token=$apiKey".
                    "&fmt=json".
                    "&symbols=".strtoupper(implode(',', $targets));
            }else if($type === 'live'){
                if(!is_array($targets)){
                    $targets = array($targets[0]);
                }
                $extraTargets = strtoupper(implode(',', array_slice($targets,1)));
                $apiUrl = "https://eodhistoricaldata.com/api/real-time/".strtoupper($targets[0]).
                    "?api_token=$apiKey".
                    "&fmt=json";
                //Extra target management.
                if($extraTargets) $apiUrl .= "&s=$extraTargets";
            }else{
                return array('error' => 'wrong type');
            }

            $tickerData = self::call_eod_api($apiUrl);
            if(!$tickerData)
                return array('error' => 'no result from real time api');

            // TODO: PLUG, remove it after realise subscribe module
            if($type === 'historical' && isset($tickerData['error_code']) && $tickerData['error_code'] === 'forbidden'){
                $tickerData = [];
                foreach ($targets as $target){
                    $apiUrl = "https://eodhistoricaldata.com/api/eod/$target?api_token=demo&fmt=json&filter=last_close";
                    $single_data = self::call_eod_api($apiUrl);
                    if($single_data && !isset($single_data['error'])){
                        $tickerData[] = [
                            'code' => explode('.', $target)[0],
                            'exchange_short_name' => explode('.', $target)[1],
                            'close' => $single_data,
                            'change' => '',
                            'change_p' => '',
                        ];
                    }
                }
            }

            return $tickerData;
        }


        /**
         * Will cal api asking then returns the result
         * @param string apiUrl
         * @return mixed
         */
        public static function call_eod_api($apiUrl)
        {
            if(!$apiUrl || gettype($apiUrl) !== 'string')
                return array('error' => 'Wrong API URL');

            //Create request and get result
            $ch = curl_init();
            curl_setopt($ch, CURLOPT_URL, $apiUrl);
            curl_setopt($ch, CURLOPT_HTTP_VERSION, CURL_HTTP_VERSION_1_1);
            curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_VERBOSE, true);
            curl_setopt($ch, CURLOPT_HEADER, true);

            $response = curl_exec($ch);

            //Parse response (headers vs body)
            $header_size = curl_getinfo($ch, CURLINFO_HEADER_SIZE);
            $headers = substr($response, 0, $header_size);
            $body = substr($response, $header_size);
            curl_close($ch);

            //Parse json body or return error
            if(!$body || strlen(trim($body)) === 0){
                return array('error_code' => 'null', 'error' => 'null body', 'headers'  => $headers);
            }
            if(in_array($body, ["Forbidden. Please contact support@eodhistoricaldata.com", "Forbidden"])){
                return array('error_code' => 'forbidden', 'error' => 'Forbidden. Perhaps your data plan does not allow you to receive data. Please contact support@eodhistoricaldata.com', 'headers'  => $headers);
            }
            if(in_array($body, ["Unauthenticated"])){
                return array('error_code' => 'unauthenticated', 'error' => 'Unauthenticated', 'headers'  => $headers);
            }

            try {
                $result = json_decode($body, true);
            } catch (Exception $err) {
                $result = array('error' => $body, 'exception' => $err->getMessage(), 'headers'  => $headers);
                error_log('Error getting api result : '.print_r($err,true));
            }

            return $result;
        }


        /**
         * Get list of news topics
         * @return array
         */
        public function get_news_topics()
        {
            return ['balance sheet', 'capital employed', 'class action', 'company announcement', 'consensus eps estimate', 'consensus estimate', 'credit rating', 'discounted cash flow', 'dividend payments', 'earnings estimate', 'earnings growth', 'earnings per share', 'earnings release', 'earnings report', 'earnings results', 'earnings surprise', 'estimate revisions', 'european regulatory news', 'financial results', 'fourth quarter', 'free cash flow', 'future cash flows', 'growth rate', 'initial public offering', 'insider ownership', 'insider transactions', 'institutional investors', 'institutional ownership', 'intrinsic value', 'market research reports', 'net income', 'operating income', 'present value', 'press releases', 'price target', 'quarterly earnings', 'quarterly results', 'ratings', 'research analysis and reports', 'return on equity', 'revenue estimates', 'revenue growth', 'roce', 'roe', 'share price', 'shareholder rights', 'shareholder', 'shares outstanding', 'strong buy', 'total revenue', 'zacks investment research', 'zacks rank'];
        }


        /**
         * Get array with hierarchy for fundamental data
         * @param string $type
         * @return array
         */
        public function get_fd_hierarchy()
        {
            return array(
                "crypto" => array(
                    "General" => array(
                        "Name","Type","Category","WebURL"
                    ),
                    "Statistics" => array(
                        "MarketCapitalization","MarketCapitalizationDiluted","CirculatingSupply","TotalSupply","MaxSupply","MarketCapDominance",
                        "TechnicalDoc","Explorer","SourceCode","MessageBoard","LowAllTime","HighAllTime"
                    ),
                ),
                "index" => array(
                    "General" => array(
                        "Code","Type","Name","Exchange","CurrencyCode","CurrencyName","CurrencySymbol","CountryName","CountryISO"
                    ),
                    "Components" => "Components",
                    "HistoricalTickerComponents" => "Historical Ticker Components",
                ),
                "fund" => array(
                    "General" => array(
                        "Code","Type","Name","Exchange","CurrencyCode","CurrencyName","CurrencySymbol","CountryName",
                        "CountryISO","ISIN","CUSIP","Fund_Summary","Fund_Family","Fund_Category","Fund_Style",
                        "Fiscal_Year_End","MarketCapitalization"
                    ),
                    "MutualFund_Data" => array(
                        "Fund_Category","Fund_Style","Nav","Prev_Close_Price","Update_Date","Portfolio_Net_Assets",
                        "Share_Class_Net_Assets","Morning_Star_Rating","Morning_Star_Risk_Rating","Morning_Star_Category",
                        "Inception_Date","Currency","Domicile","Yield","Yield_YTD","Yield_1Year_YTD","Yield_3Year_YTD",
                        "Yield_5Year_YTD","Expense_Ratio","Expense_Ratio_Date","Asset_Allocation","Value_Growth",
                        "Top_Holdings","Market_Capitalization","Top_Countries",
                        "Sector_Weights" => array("Cyclical","Defensive","Sensitive"),
                        "World_Regions" => array("Americas","Greater Asia","Greater Europe","Market Classification"),
                    )
                ),
                "etf" => array(
                    "General" => array(
                        "Code","Type","Name","Exchange","CurrencyCode","CurrencyName","CurrencySymbol","CountryName",
                        "CountryISO","Description","Category","UpdatedAt"
                    ),
                    "Technicals" => array(
                        "Beta","52WeekHigh","52WeekLow","50DayMA","200DayMA"
                    ),
                    "ETF_Data" => array(
                        "ISIN","Company_Name","Company_URL","ETF_URL","Domicile","Index_Name","Yield",
                        "Dividend_Paying_Frequency","Inception_Date","Max_Annual_Mgmt_Charge","Ongoing_Charge",
                        "Date_Ongoing_Charge","NetExpenseRatio","AnnualHoldingsTurnover","TotalAssets","Average_Mkt_Cap_Mil",
                        "Market_Capitalisation" => array("Mega","Big","Medium","Small","Micro"),
                        "Asset_Allocation","World_Regions","Sector_Weights","Fixed_Income","Holdings_Count",
                        "Top_10_Holdings","Holdings","MorningStar",
                        "Valuations_Growth" => array(
                            "Valuations_Rates_Portfolio","Valuations_Rates_To_Category","Growth_Rates_Portfolio","Growth_Rates_To_Category"
                        ),
                        "Performance" => array(
                            "1y_Volatility","3y_Volatility","3y_ExpReturn","3y_SharpRatio","Returns_YTD",
                            "Returns_1Y","Returns_3Y","Returns_5Y","Returns_10Y"
                        )
                    ),

                ),
                "common_stock" => array(
                    "General" => array(
                        "Code","Type","Name","Exchange","CurrencyCode","CurrencyName","CurrencySymbol","CountryName",
                        "CountryISO","ISIN","CUSIP","CIK","EmployerIdNumber","FiscalYearEnd","IPODate",
                        "InternationalDomestic","Sector","Industry","GicSector","GicGroup","GicIndustry","GicSubIndustry",
                        "HomeCategory","IsDelisted","Description","Address",
                        "AddressData" => array("Street","City","State","Country","ZIP"),
                        "Listings","Officers","Phone","WebURL","LogoURL","FullTimeEmployees","UpdatedAt"
                    ),
                    "Highlights" => array(
                        "MarketCapitalization","MarketCapitalizationMln","EBITDA","PERatio","PEGRatio",
                        "WallStreetTargetPrice","BookValue","DividendShare","DividendYield","EarningsShare",
                        "EPSEstimateCurrentYear","EPSEstimateNextYear","EPSEstimateNextQuarter","EPSEstimateCurrentQuarter",
                        "MostRecentQuarter","ProfitMargin","OperatingMarginTTM","ReturnOnAssetsTTM","ReturnOnEquityTTM",
                        "RevenueTTM","RevenuePerShareTTM","QuarterlyRevenueGrowthYOY","GrossProfitTTM","DilutedEpsTTM",
                        "QuarterlyEarningsGrowthYOY"
                    ),
                    "Valuation" => array(
                        "TrailingPE","ForwardPE","PriceSalesTTM","PriceBookMRQ","EnterpriseValue",
                        "EnterpriseValueRevenue","EnterpriseValueEbitda"
                    ),
                    "SharesStats" => array(
                        "SharesOutstanding","SharesFloat","PercentInsiders","PercentInstitutions","SharesShort",
                        "SharesShortPriorMonth","ShortRatio","ShortPercentOutstanding","ShortPercentFloat"
                    ),
                    "Technicals" => array(
                        "Beta","52WeekHigh","52WeekLow","50DayMA","200DayMA","SharesShort","SharesShortPriorMonth",
                        "ShortRatio","ShortPercent"
                    ),
                    "SplitsDividends" => array(
                        "ForwardAnnualDividendRate","ForwardAnnualDividendYield","PayoutRatio","DividendDate",
                        "ExDividendDate","LastSplitFactor","LastSplitDate","NumberDividendsByYear"
                    ),
                    "AnalystRatings" => array("Rating","TargetPrice","StrongBuy","Buy","Hold","Sell","StrongSell"),
                    "Holders" => array("Institutions","Funds"),
                    "InsiderTransactions",
                    "ESGScores" => array(
                        "Disclaimer","RatingDate","TotalEsg","TotalEsgPercentile","EnvironmentScore","EnvironmentScorePercentile",
                        "SocialScorePercentile","GovernanceScore","GovernanceScorePercentile","ControversyLevel","ActivitiesInvolvement"
                    ),
                )
            );
        }

        /**
         * Get array of fundamental data titles
         * @return array
         */
        public function get_fd_titles()
        {
            return array(
                "General" => "General",
                "General->Code" => "Code",
                "General->Type" => "Type",
                "General->Name" => "Name",
                "General->Exchange" => "Exchange",
                "General->CurrencyCode" => "Currency Code",
                "General->CurrencyName" => "Currency Name",
                "General->CurrencySymbol" => "Currency Symbol",
                "General->CountryName" => "Country Name",
                "General->CountryISO" => "Country ISO",
                "General->ISIN" => "ISIN",
                "General->CUSIP" => "CUSIP",
                "General->CIK" => "CIK",
                "General->EmployerIdNumber" => "Employer Id Number",
                "General->FiscalYearEnd" => "Fiscal Year End",
                "General->IPODate" => "IPO Date",
                "General->InternationalDomestic" => "International Domestic",
                "General->Sector" => "Sector",
                "General->Industry" => "Industry",
                "General->GicSector" => "Gic Sector",
                "General->GicGroup" => "Gic Group",
                "General->GicIndustry" => "Gic Industry",
                "General->GicSubIndustry" => "Gic Sub Industry",
                "General->HomeCategory" => "Home Category",
                "General->IsDelisted" => "Is Delisted",
                "General->Description" => "Description",
                "General->Address" => "Address",
                "General->AddressData" => "Address Data",
                "General->AddressData->Street" => "Street",
                "General->AddressData->City" => "City",
                "General->AddressData->State" => "State",
                "General->AddressData->Country" => "Country",
                "General->AddressData->ZIP" => "ZIP",
                "General->Listings" => "Listings",
                "General->Officers" => "Officers",
                "General->Phone" => "Phone",
                "General->WebURL" => "WebURL",
                "General->LogoURL" => "LogoURL",
                "General->Fund_Summary" => "Fund Summary",
                "General->Fund_Family" => "Fund Family",
                "General->Fund_Category" => "Fund Category",
                "General->Fund_Style" => "Fund Style",
                "General->Fiscal_Year_End" => "Fiscal Year End",
                "General->MarketCapitalization" => "Market Capitalization",
                "General->FullTimeEmployees" => "Full Time Employees",
                "General->Category" => "Category",
                "General->UpdatedAt" => "Updated At",
                "ETF_Data" => "ETF Data",
                "ETF_Data->ISIN" => "ISIN",
                "ETF_Data->Company_Name" => "Company Name",
                "ETF_Data->Company_URL" => "Company URL",
                "ETF_Data->ETF_URL" => "ETF URL",
                "ETF_Data->Domicile" => "Domicile",
                "ETF_Data->Index_Name" => "Index Name",
                "ETF_Data->Yield" => "Yield",
                "ETF_Data->Dividend_Paying_Frequency" => "Dividend Paying Frequency",
                "ETF_Data->Inception_Date" => "Inception Date",
                "ETF_Data->Max_Annual_Mgmt_Charge" => "Max Annual Mgmt Charge",
                "ETF_Data->Ongoing_Charge" => "Ongoing Charge",
                "ETF_Data->Date_Ongoing_Charge" => "Date Ongoing Charge",
                "ETF_Data->NetExpenseRatio" => "Net Expense Ratio",
                "ETF_Data->AnnualHoldingsTurnover" => "Annual Holdings Turnover",
                "ETF_Data->TotalAssets" => "TotalAssets",
                "ETF_Data->Average_Mkt_Cap_Mil" => "Average Mkt Cap Mil",
                "ETF_Data->Market_Capitalisation" => "Market Capitalisation",
                "ETF_Data->Market_Capitalisation->Mega" => "Mega",
                "ETF_Data->Market_Capitalisation->Big" => "Big",
                "ETF_Data->Market_Capitalisation->Medium" => "Medium",
                "ETF_Data->Market_Capitalisation->Small" => "Small",
                "ETF_Data->Market_Capitalisation->Micro" => "Micro",
                "ETF_Data->Asset_Allocation" => "Asset Allocation",
                "ETF_Data->World_Regions" => "World Regions",
                "ETF_Data->Sector_Weights" => "Sector Weights",
                "ETF_Data->Fixed_Income" => "Fixed Income",
                "ETF_Data->Holdings_Count" => "Holdings Count",
                "ETF_Data->Top_10_Holdings" => "Top 10 Holdings",
                "ETF_Data->Holdings" => "Holdings",
                "ETF_Data->MorningStar" => "Morning Star",
                "ETF_Data->Valuations_Growth" => "Valuations Growth",
                "ETF_Data->Valuations_Growth->Valuations_Rates_Portfolio" => "Valuations Rates Portfolio",
                "ETF_Data->Valuations_Growth->Valuations_Rates_To_Category" => "Valuations Rates To Category",
                "ETF_Data->Valuations_Growth->Growth_Rates_Portfolio" => "Growth Rates Portfolio",
                "ETF_Data->Valuations_Growth->Growth_Rates_To_Category" => "Growth Rates To Category",
                "ETF_Data->Performance" => "Performance",
                "ETF_Data->Performance->1y_Volatility" => "1y Volatility",
                "ETF_Data->Performance->3y_Volatility" => "3y Volatility",
                "ETF_Data->Performance->3y_ExpReturn" => "3y ExpReturn",
                "ETF_Data->Performance->3y_SharpRatio" => "3y SharpRatio",
                "ETF_Data->Performance->Returns_YTD" => "Returns YTD",
                "ETF_Data->Performance->Returns_1Y" => "Returns 1Y",
                "ETF_Data->Performance->Returns_3Y" => "Returns 3Y",
                "ETF_Data->Performance->Returns_5Y" => "Returns 5Y",
                "ETF_Data->Performance->Returns_10Y" => "Returns 10Y",
                "Highlights" => "Highlights",
                "Highlights->MarketCapitalization" => "Market Capitalization",
                "Highlights->MarketCapitalizationMln" => "Market Capitalization Mln",
                "Highlights->EBITDA" => "EBITDA",
                "Highlights->PERatio" => "PE Ratio",
                "Highlights->PEGRatio" => "PEG Ratio",
                "Highlights->WallStreetTargetPrice" => "Wall Street Target Price",
                "Highlights->BookValue" => "Book Value",
                "Highlights->DividendShare" => "Dividend Share",
                "Highlights->DividendYield" => "Dividend Yield",
                "Highlights->EarningsShare" => "Earnings Share",
                "Highlights->EPSEstimateCurrentYear" => "EPS Estimate Current Year",
                "Highlights->EPSEstimateNextYear" => "EPS Estimate Next Year",
                "Highlights->EPSEstimateNextQuarter" => "EPS Estimate Next Quarter",
                "Highlights->EPSEstimateCurrentQuarter" => "EPS Estimate Current Quarter",
                "Highlights->MostRecentQuarter" => "Most Recent Quarter",
                "Highlights->ProfitMargin" => "Profit Margin",
                "Highlights->OperatingMarginTTM" => "Operating Margin TTM",
                "Highlights->ReturnOnAssetsTTM" => "Return On Assets TTM",
                "Highlights->ReturnOnEquityTTM" => "Return On Equity TTM",
                "Highlights->RevenueTTM" => "RevenueTTM",
                "Highlights->RevenuePerShareTTM" => "Revenue Per Share TTM",
                "Highlights->QuarterlyRevenueGrowthYOY" => "Quarterly Revenue Growth YOY",
                "Highlights->GrossProfitTTM" => "Gross Profit TTM",
                "Highlights->DilutedEpsTTM" => "Diluted Eps TTM",
                "Highlights->QuarterlyEarningsGrowthYOY" => "Quarterly Earnings Growth YOY",
                "Valuation" => "Valuation",
                "Valuation->TrailingPE" => "Trailing PE",
                "Valuation->ForwardPE" => "Forward PE",
                "Valuation->PriceSalesTTM" => "Price Sales TTM",
                "Valuation->PriceBookMRQ" => "Price Book MRQ",
                "Valuation->EnterpriseValue" => "Enterprise Value",
                "Valuation->EnterpriseValueRevenue" => "Enterprise Value Revenue",
                "Valuation->EnterpriseValueEbitda" => "Enterprise ValueEbitda",
                "SharesStats" => "Shares Stats",
                "SharesStats->SharesOutstanding" => "Shares Outstanding",
                "SharesStats->SharesFloat" => "Shares Float",
                "SharesStats->PercentInsiders" => "Percent Insiders",
                "SharesStats->PercentInstitutions" => "Percent Institutions",
                "SharesStats->SharesShort" => "Shares Short",
                "SharesStats->SharesShortPriorMonth" => "Shares Short Prior Month",
                "SharesStats->ShortRatio" => "Short Ratio",
                "SharesStats->ShortPercentOutstanding" => "Short Percent Outstanding",
                "SharesStats->ShortPercentFloat" => "Short Percent Float",
                "Technicals" => "Technicals",
                "Technicals->Beta" => "Beta",
                "Technicals->52WeekHigh" => "52 Week High",
                "Technicals->52WeekLow" => "52 Week Low",
                "Technicals->50DayMA" => "50 Day MA",
                "Technicals->200DayMA" => "200 Day MA",
                "Technicals->SharesShort" => "Shares Short",
                "Technicals->SharesShortPriorMonth" => "Shares Short Prior Month",
                "Technicals->ShortRatio" => "Short Ratio",
                "Technicals->ShortPercent" => "Short Percent",
                "SplitsDividends" => "SplitsDividends",
                "SplitsDividends->ForwardAnnualDividendRate" => "Forward Annual Dividend Rate",
                "SplitsDividends->ForwardAnnualDividendYield" => "Forward Annual Dividend Yield",
                "SplitsDividends->PayoutRatio" => "Payout Ratio",
                "SplitsDividends->DividendDate" => "Dividend Date",
                "SplitsDividends->ExDividendDate" => "Ex Dividend Date",
                "SplitsDividends->LastSplitFactor" => "Last Split Factor",
                "SplitsDividends->LastSplitDate" => "Last Split Date",
                "SplitsDividends->NumberDividendsByYear" => "Number Dividends By Year",
                "AnalystRatings" => "AnalystRatings",
                "AnalystRatings->Rating" => "Rating",
                "AnalystRatings->TargetPrice" => "Target Price",
                "AnalystRatings->StrongBuy" => "Strong Buy",
                "AnalystRatings->Buy" => "Buy",
                "AnalystRatings->Hold" => "Hold",
                "AnalystRatings->Sell" => "Sell",
                "AnalystRatings->StrongSell" => "StrongSell",
                "Holders" => "Holders",
                "Holders->Institutions" => "Institutions",
                "Holders->Funds" => "Funds",
                "InsiderTransactions" => "Insider Transactions",
                "ESGScores" => "ESG Scores",
                "ESGScores->Disclaimer" => "Disclaimer",
                "ESGScores->RatingDate" => "Rating Date",
                "ESGScores->TotalEsg" => "Total ESG",
                "ESGScores->TotalEsgPercentile" => "Total ESG Percentile",
                "ESGScores->EnvironmentScore" => "Environment Score",
                "ESGScores->EnvironmentScorePercentile" => "Environment Score Percentile",
                "ESGScores->SocialScorePercentile" => "Social Score Percentile",
                "ESGScores->GovernanceScore" => "Governance Score",
                "ESGScores->GovernanceScorePercentile" => "Governance Score Percentile",
                "ESGScores->ControversyLevel" => "Controversy Level",
                "ESGScores->ActivitiesInvolvement" => "Activities Involvement",
                "Statistics->MarketCapitalization" => "Market Capitalization",
                "Statistics->MarketCapitalizationDiluted" => "Market Capitalization Diluted",
                "Statistics->CirculatingSupply" => "Circulating Supply",
                "Statistics->TotalSupply" => "Total Supply",
                "Statistics->MaxSupply" => "Max Supply",
                "Statistics->MarketCapDominance" => "Market Cap Dominance",
                "Statistics->TechnicalDoc" => "Technical Doc",
                "Statistics->Explorer" => "Explorer",
                "Statistics->SourceCode" => "Source Code",
                "Statistics->MessageBoard" => "Message Board",
                "Statistics->LowAllTime" => "Low All Time",
                "Statistics->HighAllTime" => "High All Time",
            );
        }

        /**
         * Get library of financials labels
         * @return array
         */
        public function get_financials_lib()
        {
            return array(
                "Earnings->History" => array(
                    "Earnings" => array(
                        "History" => array(
                            "_timeline_History" => "quarterly",
                            "beforeAfterMarket" => "Before After Market",
                            "currency" => "Currency",
                            "epsActual" => "EPS Actual",
                            "epsEstimate" => "EPS Estimate",
                            "epsDifference" => "EPS Difference",
                            "surprisePercent" => "Surprise Percent"
                        ),
                    ),
                ),
                "Earnings->Trend" => array(
                    "Earnings" => array(
                        "Trend" => array(
                            "_timeline_Trend" => "quarterly",
                            "period" => "Period",
                            "growth" => "Growth",
                            "earningsEstimateAvg" => "Earnings Estimate Avg",
                            "earningsEstimateLow" => "Earnings Estimate Low",
                            "earningsEstimateHigh" => "Earnings Estimate High",
                            "earningsEstimateYearAgoEps" => "Earnings Estimate Year Ago EPS",
                            "earningsEstimateNumberOfAnalysts" => "Earnings Estimate Number of Analysts",
                            "earningsEstimateGrowth" => "Earnings Estimate Growth",
                            "revenueEstimateAvg" => "Revenue Estimate Avg",
                            "revenueEstimateLow" => "Revenue Estimate Low",
                            "revenueEstimateHigh" => "Revenue Estimate High",
                            "revenueEstimateYearAgoEps" => "Revenue Estimate Year Ago EPS",
                            "revenueEstimateNumberOfAnalysts" => "Revenue Estimate Number of Analysts",
                            "revenueEstimateGrowth" => "Revenue Estimate Growth",
                            "epsTrendCurrent" => "EPS Trend Current",
                            "epsTrend7daysAgo" => "EPS Trend 7 days Ago",
                            "epsTrend30daysAgo" => "EPS Trend 30 days Ago",
                            "epsTrend60daysAgo" => "EPS Trend 60 days Ago",
                            "epsTrend90daysAgo" => "EPS Trend 90 days Ago",
                            "epsRevisionsUpLast7days" => "EPS Revisions up Last 7 days",
                            "epsRevisionsUpLast30days" => "EPS Revisions up Last 30 days",
                            "epsRevisionsDownLast7days" => "EPS Revisions down Last 7 days",
                            "epsRevisionsDownLast30days" => "EPS Revisions down Last 30 days"
                        ),
                    ),
                ),
                "Earnings->Annual" => array(
                    "Earnings" => array(
                        "Annual" => array(
                            "_timeline_Annual" => "yearly",
                            "epsActual" => "EPS Actual"
                        )
                    ),
                ),
                "Financials->Balance_Sheet" => array(
                    "Financials" => array(
                        "Balance_Sheet" => array(
                            "_timeline_Balance_Sheet" => "both",
                            "currency_symbol" => "Currency Symbol",
                            "totalAssets" => "Total Assets",
                            "intangibleAssets" => "Intangible Assets",
                            "earningAssets" => "Earning Assets",
                            "otherCurrentAssets" => "Other Current Assets",
                            "totalLiab" => "Total Liab",
                            "totalStockholderEquity" => "Total Stockholder Equity",
                            "deferredLongTermLiab" => "Deferred Long Term Liab",
                            "otherCurrentLiab" => "Other Current Liab",
                            "commonStock" => "Common Stock",
                            "retainedEarnings" => "Retained Earnings",
                            "otherLiab" => "Other Liab",
                            "goodWill" => "Good Will",
                            "otherAssets" => "Other Assets",
                            "cash" => "Cash",
                            "totalCurrentLiabilities" => "Total Current Liabilities",
                            "netDebt" => "Net Debt",
                            "shortTermDebt" => "Short Term Debt",
                            "shortLongTermDebt" => "Short Long Term Debt",
                            "shortLongTermDebtTotal" => "Short Long Term Debt Total",
                            "otherStockholderEquity" => "Other Stockholder Equity",
                            "propertyPlantEquipment" => "Property Plant Equipment",
                            "totalCurrentAssets" => "Total Current Assets",
                            "longTermInvestments" => "Long Term Investments",
                            "netTangibleAssets" => "Net Tangible Assets",
                            "shortTermInvestments" => "Short Term Investments",
                            "netReceivables" => "Net Receivables",
                            "longTermDebt" => "Long TermDebt",
                            "inventory" => "Inventory",
                            "accountsPayable" => "Accounts Payable",
                            "totalPermanentEquity" => "Total Permanent Equity",
                            "noncontrollingInterestInConsolidatedEntity" => "Non Controlling Interest In Consolidated Entity",
                            "temporaryEquityRedeemableNoncontrollingInterests" => "Temporary Equity Redeemable Non Controlling Interests",
                            "accumulatedOtherComprehensiveIncome" => "Accumulated Other Comprehensive Income",
                            "additionalPaidInCapital" => "Additional Paid In Capital",
                            "commonStockTotalEquity" => "Common Stock Total Equity",
                            "preferredStockTotalEquity" => "Preferred Stock Total Equity",
                            "retainedEarningsTotalEquity" => "Retained Earnings Total Equity",
                            "treasuryStock" => "Treasury Stock",
                            "accumulatedAmortization" => "Accumulated Amortization",
                            "nonCurrrentAssetsOther" => "Non Current Assets Other",
                            "deferredLongTermAssetCharges" => "Deferred Long Term Asset Charges",
                            "nonCurrentAssetsTotal" => "Non Current Assets Total",
                            "capitalLeaseObligations" => "Capital Lease Obligations",
                            "longTermDebtTotal" => "Long Term Debt Total",
                            "nonCurrentLiabilitiesOther" => "Non Current Liabilities Other",
                            "nonCurrentLiabilitiesTotal" => "Non Current Liabilities Total",
                            "negativeGoodwill" => "Negative Goodwill",
                            "warrants" => "Warrants",
                            "preferredStockRedeemable" => "Preferred Stock Redeemable",
                            "capitalSurpluse" => "Capital Surplus",
                            "liabilitiesAndStockholdersEquity" => "Liabilities And Stockholders Equity",
                            "cashAndShortTermInvestments" => "Cash AndShort Term Investments",
                            "propertyPlantAndEquipmentGross" => "Property Plant And Equipment Gross",
                            "propertyPlantAndEquipmentNet" => "Property Plant And Equipment Net",
                            "accumulatedDepreciation" => "Accumulated Depreciation",
                            "netWorkingCapital" => "Net Working Capital",
                            "netInvestedCapital" => "Net Invested Capital",
                            "commonStockSharesOutstanding" => "Common Stock Shares Outstanding",
                        ),
                    ),
                ),
                "Financials->Cash_Flow" => array(
                    "Financials" => array(
                        "Cash_Flow" => array(
                            "_timeline_Cash_Flow" => "both",
                            "currency_symbol" => "Currency Symbol",
                            "investments" => "Investments",
                            "changeToLiabilities" => "Change to Liabilities",
                            "totalCashflowsFromInvestingActivities" => "Total Cash Flows From Investing Activities",
                            "netBorrowings" => "Net Borrowings",
                            "totalCashFromFinancingActivities" => "Total Cash from Financing Activities",
                            "changeToOperatingActivities" => "Change to Operating Activities",
                            "netIncome" => "Net Income",
                            "changeInCash" => "Change in Cash",
                            "beginPeriodCashFlow" => "Begin Period Cash Flow",
                            "endPeriodCashFlow" => "End Period Cash Flow",
                            "totalCashFromOperatingActivities" => "Total Cash From Operating Activities",
                            "depreciation" => "Depreciation",
                            "otherCashflowsFromInvestingActivities" => "Other Cash Flows from Investing Activities",
                            "dividendsPaid" => "Dividends Paid",
                            "changeToInventory" => "Change to Inventory",
                            "changeToAccountReceivables" => "Change to Account Receivables",
                            "salePurchaseOfStock" => "Sale Purchase of Stock",
                            "otherCashflowsFromFinancingActivities" => "Other Cash Flows from Financing Activities",
                            "changeToNetincome" => "Change to Net Income",
                            "capitalExpenditures" => "Capital Expenditures",
                            "changeReceivables" => "Change Receivables",
                            "cashFlowsOtherOperating" => "Cash Flows Other Operating",
                            "exchangeRateChanges" => "Exchange Rate Changes",
                            "cashAndCashEquivalentsChanges" => "Cash and Cash Equivalents Changes",
                            "changeInWorkingCapital" => "Change in Working Capital",
                            "otherNonCashItems" => "Other Non Cash Items",
                            "freeCashFlow" => "Free Cash Flow",
                        ),
                    ),
                ),
                "Financials->Income_Statement" => array(
                    "Financials" => array(
                        "Income_Statement" => array(
                            "_timeline_Income_Statement" => "both",
                            "currency_symbol" => "Currency Symbol",
                            "researchDevelopment" => "Research Development",
                            "effectOfAccountingCharges" => "Effect of Accounting Charges",
                            "incomeBeforeTax" => "Income Before Tax",
                            "minorityInterest" => "Minority Interest",
                            "netIncome" => "Net Income",
                            "sellingGeneralAdministrative" => "Selling General Administrative",
                            "sellingAndMarketingExpenses" => "Selling and Marketing Expenses",
                            "grossProfit" => "Gross Profit",
                            "reconciledDepreciation" => "Reconciled Depreciation",
                            "ebit" => "EBIT",
                            "ebitda" => "EBITDA",
                            "depreciationAndAmortization" => "Depreciation and Amortization",
                            "nonOperatingIncomeNetOther" => "Non Operating Income Net Other",
                            "operatingIncome" => "Operating Income",
                            "otherOperatingExpenses" => "Other Operating Expenses",
                            "interestExpense" => "Interest Expense",
                            "taxProvision" => "Tax Provision",
                            "interestIncome" => "Interest Income",
                            "netInterestIncome" => "Net Interest Income",
                            "extraordinaryItems" => "Extraordinary Items",
                            "nonRecurring" => "Non Recurring",
                            "otherItems" => "Other Items",
                            "incomeTaxExpense" => "Income Tax Expense",
                            "totalRevenue" => "Total Revenue",
                            "totalOperatingExpenses" => "Total Operating Expenses",
                            "costOfRevenue" => "Cost of Revenue",
                            "totalOtherIncomeExpenseNet" => "Total Other Income Expense Net",
                            "discontinuedOperations" => "Discontinued Operations",
                            "netIncomeFromContinuingOps" => "Net Income From Continuing Ops",
                            "netIncomeApplicableToCommonShares" => "Net Income Applicable to Common Shares",
                            "preferredStockAndOtherAdjustments" => "Preferred Stock and Other Adjustments",
                        )
                    )
                )
            );
        }



        //$cc_json_raw = json_decode( file_get_contents('https://eodhistoricaldata.com/api/exchange-symbol-list/CC?api_token=&fmt=json') );
        //$forex_json_raw = json_decode( file_get_contents('https://eodhistoricaldata.com/api/exchange-symbol-list/FOREX?api_token=&fmt=json') );
        //
        //$cc_list = [];
        //$forex_list = [];
        //foreach ($cc_json_raw as $cc_item){
        //    $path = explode( '-', $cc_item->Code );
        //    if( end($path) === 'USD' ){
        //        $cc_list[] = substr($cc_item->Code, 0, -4);
        //    }
        //}
        //foreach ($forex_json_raw as $forex_item){
        //    if( strlen($forex_item->Code) === 3 ){
        //        $forex_list[] = $forex_item->Code;
        //    }
        //}

        /**
         * Get array of cryptocurrency codes
         * @return array
         */
        public function get_cc_codes()
        {
            return ['$ANRX','$KING','$TRDL','0xBTC','1GOLD','1INCH','1MT','1WO','1X2','2GIVE','2KEY','3X-LONG-BITCOIN-TOKEN','42','4ART','777','7ADD','808TA','A','A5T','AAA','AAB','AAC','AAPL','AAPL1','AAVE','ABAT','ABBC','ABCC-TOKEN','Abet','ABL','ABNB','ABS','ABT','ABUSD','ABX','ABYSS','AC','ACA1','ACAT','ACDC','ACES','ACH1','ACM','ACOIN','ACS','ACT','ACTINIUM','ADA','ADABEAR','ADABULL','ADADOWN','ADAI','ADAUP','ADAX','ADB','ADC','ADEL','ADI','ADK','ADM','ADP','ADS','ADX','ADZ','AE','aENJ','AENS','AEON','AERGO','AET','aEth','AFC','AFIN','AfroX','AGIX','AGLD','AGVC','AIB','AIDOC','AIM','AIN','AION','AIT','AITRA','AK12','AKA','AKITA','AKN','AKRO','AKT','ALAYA','ALBT','ALCHEMY','ALD','ALEPH','ALG','ALGO','ALI','ALIAS','ALIBABA-TOKENIZED-STOCK-BITTREX','ALICE','ALINK','ALLBI','ALOHA','ALPA','ALPACA','ALPHA','ALTBULL','ALTCOMMUNITY-COIN','ALY','AMA','AMAZON-TOKENIZED-STOCK-FTX','AMB','AMC','AMD','AME','AMKR','AMLT','AMM','AMN','AMO','AMON','AMP','AMPL','AMS','ANC','ANC1','ANCHOR','ANCT','ANG','ANI','ANKR','ANON','ANT','ANTISCAMTOKEN','ANW','ANY','AOA','AOG','APC','APE','APE3','API3','APIX','APL','APM','APOLLON-LIMASSOL','APPC','APPLE-TOKENIZED-STOCK-FTX','APY','APY-VISION','AQT','AR','ARAW','ARB','ARCH','ARCO','ARCONA','ARCT','ARDR','ARDX','AREPA','ARGUS','ARIA20','ARION','ARK','ARKK','ARMOR','ARNX','arNXM','ARO','ARPA','ARQ','ARRR','ART','ARX','ASAFE','ASK','ASKO','ASM','ASNX','ASP','ASR','AST','ASTA','ASTRO1','ASY','ATA','ATB','ATL','ATM1','ATMOS','ATOM','ATOMBULL','ATP','ATRI','ATT','ATTN','ATUSD','AUC','AUCTION','AUDIO','AUDT','AUR','AURORA','AURORA1','AUSCM','AUTO1','AUTOFARM','AUX','AV','AVA','AVAX','AVC','AVT','AWC','AWG','AWX','AXE','AXEL','AXI','AXIA','AXIOM','AXIS','AXL','AXPR','AXS','AYA','AZ','AZUKI','B1P','B20','B21','B2M','BABA','BAC','BADGER','BAKE','BAL','BALI','BAMBOO','BAN','BANANA','BANCA','BAND','BANKROLL-VAULT','BAO','BAR','BART','BAS','BASE','BASIC','BASID','BASIS-DOLLAR','BASX','BAT','BAX','BB','BBGC','BBI','BBK','BBO','BBP','BBS','BC','BCA','BCD','BCDN','BCEO','BCH','BCHBEAR','BCHBULL','BCHC','BCN','BCNA','BCNT','BCP','BCPT','BCV','BCX','BCZ','BDCASH','BDCC','BDO','BDP','BDT','BDX','BEAM','BEAR','BEEFY-FINANCE','BEET','BEETS','BEL','BELA','BEPRO','BERN','BERRY','BEST','BETA','BETA1','BETH','BF','BFF','BFLY','BFT','BGL','BHC','BHD','BHIG','BHP','BID','BIDAO','BIDR','BIFI','BIGONE-TOKEN','BILI','BILLIBILLI-TOKENIZED-STOCK-BITTREX','BIONTECH-TOKENIZED-STOCK-BITTREX','BIONTECH-TOKENIZED-STOCK-FTX','BIP','BIR','BIRDCHAIN','BIS','BIT','BIT1','BITALGO','BITB','BITC','BITCI','BITCNY','BITCOIN-CLASSIC','BITCOIN-FILE','BITCOINV','BITCOIVA','BITETHER','BITO','BITPAYER-TOKEN','BITS','BITSTAR','BITSTEN-TOKEN','BITT','BITX','BIX','BIZZ','BKBT','BKK','BKKG','BLACK','BLAST','BLAZR','BLCT','BLINK','BLK','BLOC','BLOCK','BLOCKCLOUD','BLOCKIDCOIN','BLOCKMASON-LINK','BLTG','BLU','BLUE-BAIKAL','BLUE','BLURT','BLV','BLX','BLY','BLZ','BMI','BMP','BMX','BNA','BNANA','BNB','BNBBULL','BNBDOWN','BNBUP','BNC','BNF','BNK','BNOX','BNS','BNSD','BNT','BNTX','BNTY','BOA','BOB','BOLI','BOLT','BOLTT','BOMB','BOND','BONDLY','BONK','BONO','BOOST','BOR','BORA','BOSON','BOTS','BOUTS','BPLC','BPP','BPS','BQT','BQTX','BRD','BREW','BRY','BRZ','BRZE','BSC','BSCGIRL','BSCX','BSD','BSDS','BSE','BSOV','BSP','BST','BST1','BSTY','BSV','BSVBEAR','BSVBULL','BSW','BSY','BT','BTA','BTAD','BTB','BTC','BTC2','BTCB','BTCDOWN','BTCHG','BTCP','BTCR','BTCST','BTCT','BTCUP','BTCV','BTCZ','BTD','BTDX','BTG','BTG1','BTM','BTMX','BTNYX','BTO','BTR','BTRL','BTRS','BTS','BTSC','BTSE','BTT','BTU','BTW','BTX','BTZC','BUB','BUIDL','BUILD','BUL','BULLSHIT','BUND','BUNNY','BUP','BURGER','BURN','BUSD','BUT','BUX','BUXCOIN','BUY','BUZZ','BVL','BVOL','BWB','BXA','BXC','BXY','BYND','BYTE','BZE','BZNT','C2','C20','C2O','C98','CAB','CAI','CAKE','CAMP','CAN','CANN','CAP','CAPITAL-FINANCE','CAPP','CARAT','CARBON','CARD','CARE','CAS','CASH','CAT','CATT','CATX','CAVO','CBANK','CBAT','CBC','CBET','CBIX','CBK','CBM','CBP','CBSE','CCA','CCE','CCO','CCX','CCXX','CDAI','CDEX','CDL','CEL','CELO','CELR','CENNZ','CERE','CET','CF','CFi','CFL','CFX','CGT','CHADS','CHAIN','CHAINPAY','CHAINZ-ARENA','CHANNELS','CHART','CHAT','CHEESE','CHESS','CHFT','CHG','CHI-GASTOKEN','CHI','CHIK','CHM','CHND','CHONK','CHP','CHR','CHS','CHSB','CHX','CHZ','CIR','CITY','CIV','CIX100','CJ','CKB','CLA','CLAM','CLB','CLBR','CLC','CLIQ','CLO','CLOAK','CLOUT','CLR','CLT','CLUB','CLV','CLX','CMIT','CMM','CMT','CNB','CND','CNFI','CNN','CNNC','CNNS','CNS','CNT','CNTM','CNTR','CNX','COAL','COCOS','COFI','COFIX','COIL','COIN','COLX','COMB','COMBINE-FINANCE','COMBO','COMET','COMP','CON','CONTENTBOX','CONX','COR','CORAL','CORE','CORN','CORNICHON','CORX','COS','COSM','COT','COTI','COV','COVA','COVAL','CPC','CPI','CPT','CPUCOIN','CQT','CRAFT','CRBN','CRD','CRDT','CRE','CREAM','CREDIT','CREP','CREX','CRM','CRO','CROAT','CRPT','CRT','CRU','CRV1','CRW','CRYPTOBHARATCOIN','CRYPTOSOUL','CS','CSC','CSP','CSPN','CSTL','CTASK','CTC','CTCN','CTI','CTK','CTL','CTRT','CTSI','CTT','CTX1','CTXC','CUBE1','CUDOS','CURE','CURIO-GOVERNANCE','CUSDC','CUSDT','CUT','CV','CVC','CVCC','CVNT','CVP','CVR','CVT','CVX','CWBTC','CWS','CWV','CXC','CXN','CXO','CXT','CYL','CYMT','CYTR','CZRX','D','D4RK','DAAPL','DAC','DACC','DAD','DAG','DAI','DAM','DAO','DAPP','DAPPT','DAPS','DAR','DARA','DARK-ENERGY-CRYSTALS','DASH','DASHG','DATA','DATP','DATX','DAV','DAWN','DAX','DAY','DBC','DBET','DCD','DCL','DCN','DCNTR','DCR','DDD','DDIM','DDK','DDR','DDRT','DDX','DEB','DECURIAN','DEEP','DEFI','DEFIAT','DEFIBOX','DEFLCT','DEFO','DEGO','DEM','DENT','DEP','DEPAY','DEQ','DERI','DERO','DETS','DEUS','DEV','DEVA','DEXA','DEXE','DEXG','DEXM','DF','DFD','DFGL','DFI','DFINANCE','DFIO','DFIP','DFL','DFT','DFX','DGB','DGC','DGD','DGP','DGTX','DGVC','DGX','DHT','DIA','DIAMOND','DIC','DICE','DIFY-FINANCE','DIGEX','DIGG','DIGIFINEXTOKEN','DIGITAL-RESERVE-CURRENCY','DILI','DIME','DIN','DION','DIP','DIPPER-NETWORK','DIS','DISTX','DIT','DITTO','DIVI','DJ15','DJV','DKA','DLC','DLT','DMCH','DMD','DMG','DML','DMME','DMS','DMST','DMT','DMTC','DMX','DNA','DNT','DOCK','DODO','DOGDEFI','DOGE','DOGEBULL','DOGEC','DOGES','DOKI','DON','DONU','DONUT','DOOS','DOS','DOT','DOTDOWN','DOTUP','DOTX','DOUGH','DOV','DOW','DPC','DPI','DPY','DRAGON-TOKEN','DRC-MOBILITY','DRC','DREAMCOIN','DREP','DRGB','DRGN','DRGNBULL','DRM','DRS','DSD','DSLA','DSS','DT','DTA','DTC','DTEP','DTH','DTX','DUCATO','DUCK','DUN','DUSD','DUSK','DVC','DVG','DVI','DVP','DVT','DX','DXF','DYDX','DYN','DYNMT','DYP','DYT','EARN-DEFI','EARN','EARNBET','EBST','EC','ECA','ECC','ECELL','ECO','ECO-VALUE-COIN','ECOIN','ECOS','ECU','EDC','EDG','EDGELESS','EDI','EDN','EDR','EDRC','EFI','EFK','EFL','EFX','EGCC','EGEM','EGG','EGLD','EGT','EHRT','EIDOS','EKO','EKT','ELA','ELD','ELEC','ELF','ELON','ELT','ELTCOIN','ELX','ELY','EM','EMC','EMC2','EMOGI-NETWORK','EMPOW','EMRALS','EMRX','EMT','eMTRG','ENCRYPGEN','ENG','ENJ','ENQ','ENT','ENTRC','ENTS','EOS','EOSBEAR','EOSBULL','EOSC','EOSDAC','EOSDT','EOST','EOX','EPAN','EPIC','EPS','EQL','EQMT','EQUAD','ERC20','ERG','ERN','erowan','ERSDL','ES','ESD','ESH','ESK','ESRC','ESS','ESTI','ETC','ETCBEAR','ETCBULL','ETG','ETGF','ETGP','ETH','ETH20SMACO','ETHBEAR','ETHBTCRSI','ETHBULL','ETHHEDGE','ETHIX','ETHM','ETHMACOAPY','ETHO','ETHPA','ETHPY','ETHRSI6040','ETHUP','ETHY','ETHYS','ETM','ETN','ETNX','ETP','ETZ','EUC','EUM','EUNO','EURS','EURU','EVC','EVE','EVED','EVIL','EVN','EVR','EVT','EVX','EVZ','EWT','EXCL','EXE','EXF','EXM','EXMR','EXNT','EXO','EXP','eXRD','EXRN','EXTRADNA','EYE','EYES','EZ','EZY','F1C','FACE','FACEBOOK-TOKENIZED-STOCK-BITTREX','FAIRGAME','FAR','FARM','FAST','FB','FBN','FBT','FCD','FCH','FCT','FDR','FDZ','FET','FEX','FFF','FFYI','FIC','FIDA','FIL','FILDA','FIN','FINANCIAL-INVESTMENT-TOKEN','FINEXBOXTOKEN','FIO','FIRE-PROTOCOL','FIRE','FIRMACHAIN','FIRO','FIS','FIT','FKX','FLEX','FLG','FLIXX','FLL','FLM1','FLO','FLOKI','FLOT','FLOW','FLS','FLX','FMA','FME','FMTA','FNK','FNSP','FNT','FNX','FOAM','FOL','FONT','FOR','FORS','FORTH','FOUR','FOX','FOXT','FRAX','FRC','FRED','FREE','FRM','FRMS','FRMX','FRN','FRONT','FRST','FS','FSC','FSCC','FSHN','FSN','FST','FSW','FTC','FTI','FTM','FTO','FTT','FTX','FTXT','FUEL','FUN','FUND','FUSE','FUZZ','FWT','FX','FX1','FXF','FXP','FXS','FXT','FYD','FYP','FYZ','G999','GAL','GALA','GAMB','GAME','GARD','GAS','GASP','GB','GBPU','GBRICK','GBTC','GBX','GBYTE','GCC','GCN','GCR','GCX','GDAO','GDC','GDR','GEAR','GEEQ','GEM','GEN','GENE','GENE1','GEO','GERA','GET','GF','GFARM2','GGTK','GHD','GHOST','GHOSTPRISM','GHST','GIO','GIV','GLC','GLCH','GLDY','GLEEC','GLM','GLMR','GLOB','GLOBAL-X-CHANGE-TOKEN','GLOX','GLS','GLT','GM','GMAT','GMC','GME','GMNG','GMT','GMT3','GNC','GNO','GNX','GNY','GO','GOD','GOF','GOGO','GOHELPFUND','GOLD-COIN-RESERVE','GOLD','GOLDEN-TOKEN','GOLDR','GOM','GOM2','GONETWORK','GOOGL','GOOGLE-TOKENIZED-STOCK-BITTREX','GOSS','GOT','GOVI','GPKR','GPYX','GRAP','GRC','GRFT','GRG','GRIC-COIN','GRIM-EVO','GRIMM','GRIN','GRLC','GRN','GRO','GRPL','GRS','GRT1','GSC','GSE','GSR','GST','GSWAP','GT','GTC','GTC2','GTF','GTH','GTM','GTO','GTX','GUAP','GUCCIONECOIN','GUM','GUSD','GVT','GXC','GXT','GXX','GYSR','GZE','GZIL','HAI','HAKKA','HALV','HAND','HANDY','HAPY','HARD','HATCH','HAVY','HB','HBAR','HBC','HBD','HBDC','HBN','HBO','HBTC-TOKEN','HBTC','HBX','HDAO','HEAT','HEDG','HEDGE','HEGIC','HELMET','HERB','HEX','HEZ','HFI','HGET','HGOLD','HH','HIBS','HIGH','HINT','HIT','HITX','HIVE','HIZ','HLIX','HMQ','HMR','HNB','HNC','HNDC','HNS','HNST','HNT','HNY','HODL1','HOGE','HOMI','HOMT','HOO','HOPR','HOT','HOT1','HPAY','HPB','HPS','HPT','HPY','HQT','HQX','HSC','HT','HTA','HTBULL','HTDF','HTML','HTN','HTR','HUB','HUM','HUNT','HUSD','HUSH','HUSL','HVCO','HVN','HXRO','HYC','HYDRA','HYDRO-PROTOCOL','HYDRO','HYN','HYVE','I9C','IBANK','IBP','IBS','IBTC','IBVOL','IC','ICE','ICH','ICHI','ICNQ','ICOB','ICON','ICP','ICX','ID','IDALL','IDEA','IDEX','IDH','IDK','IDLE','IDNA','IDT','IDX','IETHEREUM','IFEX','IFLT','IFT','IFX24','IG','IGNIS','IHT','IIC','ILC','ILK','ILV','imBTC','IMG','IMGC','IMPL','IMS','IMSMART','IMT','IMX','IMX1','IND','INDEX','INFI','INFS','INJ','INK','INN','INNBC','INSTAR','INSUR','INT','INTRATIO','INTX','INVE','INXT','IOG','ION','IONC','IOST','IOTX','IPL','IPM','IPX','IQ','IQCASH','IQN','IRA','IRD','IRIS','ISAL','ISDT','ISIKC','ISR','ITAM','ITC','ITEN','ITL','ITS','IUT','IX','IXC','IXT','IZE','JADE','JAR','JASMY','JBX','JCC','JDC','JEM','JET','JEWEL','JFC','JFI','JGN','JMC','JNTR','JOINT','JOYS','JRT','JS','JSB','JST','JT','JULD','JUP','JUV','JWL','KAI','KAN','KANGAL','KARMA','KASH','KAT','KAU','KAVA','KBC','KCASH','KCS','KDA','KDAG','KDG','KEEP','KEMA','KEN','KEX','KEY','KEYFI','KEYT','KFC','KFX','KGC','KICKS','KIF','KIM','KIMCHI','KIN','KIND','KISHU','KIT','KLAY','KLIMA','KLKS','KLP','KLV','KMD','KMW','KNC','KNDC','KNT','Kobe','KOBO','KODX','KOIN','KOK','KP2R','KP3R','KP3RB','KP4R','KRB','KRL','KRT','KSEED','KSM','KSP','KSS','KT','KTLYO','KTON','KTS','KTT','KUJI','KURT','KUV','KVA','KWATT','KXC','KYSC','KZC','L2','L2P','LA','LAMB','LANA','LAR','LAVA','LAYER','LAZIO','LBA','LBC','LBK','LBTC','LBXC','LBY','LCC','LCG','LCP','LCS','LCX','LDN','LDO','LEAD','LEGALBLOCK','LEMO','LEO','LET','LEVELG','LGCY','LHB','LHT','LIB','LIBARTYSHARETOKEN','LIBFX','LIBREF','LIEN','LIKE','LIMEX','LIMIT','LINA','LINA1','LINEAR','LINK','LINKA','LINKBEAR','LINKBULL','LINKDOWN','LINKETHPA','LINKETHRSI','LINKPT','LINKTOKEN','LINKUP','LIQUID','LIT','LIT1','LITION','LIVE','LKK','LKN','LMCH','LML','LMY','LN','LNC','LND','LNOT','LNT','LOA','LOC','LOCK','LOG','LOL','LON','LOOKS','LOOM','LOON','LOOT','LOTTO','LOV','LOVE','LPOOL','LPT','LQDR','LRC','LRG','LSK','LSV','LTC','LTCBEAR','LTCBULL','LTCR','LTCU','LTHN','LTK','LTO','LTX','LUA','LUD','LUN','LUNA1','LUNA2','LUNES','LUX','LXT','LYFE','LYM','LYNC','LYR','LYXe','MAC','MACH','MAHA','MAID','MAN','MANA','MANDI','MANNA','MAP','MAPR','MAPS','MAR','MARO','MARSCOIN','MARTK','MAS','MASK','MASQ','MASS','MATH','MATIC','MATICBULL','MATTER','MAX-EXCHANGE-TOKEN','MAX','MAY','MBC','MBL','MBN','MBOX','MBX','MC','MCAN','MCB','MCH','MCI','MCM','MCOBIT','MCPC','MDA','MDM','MDO','MDS','MDT','MDU','MDX','MEC','MED','MEDI','MEDIBIT','MEDIC','MEET','MEETONE','MEGA','MEMBRANA','MEME','MEMETIC','MERI','MES','MET','META','METIS','MEX','MFG','MFT','MGC','MHC','MIB','MIC','MICRO','MIDAS','MIDBULL','MIKS','MILEVERSE','MILK2','MIN','MINA','MINE','MINI','MINT','MINTCOIN','MINTME','MIOTA','MIR-COIN','MIR','MIRROR','MIS','MITH','MITX','MIX','MIXTRUST','MKCY','MKR','MLGC','MLK','MLN','MLR','MM','MMO','MNC','MNE','MNR','MNS','MNTIS','MNTP','MOB','MOBI','MOC','MODEX','MODIC','MODX','MOF','MOGX','MOJO','MOLK','MONA','MONAVALE','MONEYNET','MOONDAY','MOONS','MORE','MORPHER','MOTA','MOTO','MOV','MOVR','MP3','MPH','MQL','MRNA','MRPH','MRX','MSB','MSR','MST','MSTR','MSWAP','MT','MTA','MTC','MTH','MTL','MTLX','MTN','MTR','MTRG','MTS','MTV','MTX','MULTI','MUSD','MUSE','MUST','MVEDA','MVL','MVP','MWC','MX','MXC','MXT','MXW','MXX','MYST','MYX','N0001','N3RDz','NAME-CHANGING-TOKEN','NAMI','NAN','NANJ','NAS','NASH','NAV','NAVI','NAX','NBC','NBOT','NBR','NBS','NBT','NBX','NBXC','NCASH','NCC','NCDT','NCP','NCT','NDAU','NDN','NDR','NDX','NEAR','NEBL','NEBO','NEO','NEST','NESTEGG-COIN','NESTREE','NETFLIX-TOKENIZED-STOCK-BITTREX','NETKO','NETKOIN','NEU','NEVA','NEW','NEX','NEXO','NFLX','NFT','NFTX-HASHMASKS-INDEX','NFTX','NFUP','NFXC','NFY','NGC','NGM','NIA','NIF','NIM','NIO','NKN','NLC2','NLG','NMC','NMR','NODE','NOIA','NOKN','NOR','NORD','NOTE','NOVA','NOW','NPLC','NPXS','NPXSXEM','NRG','NRP','NSBT','NSD','NST','NSURE','NTB','NTK','NTR','NTRN','NU','NULS','NUT','NUTS','NUX','NVA','NVDA','NVT','NWC','NXM','NXS','NXT','NYAN-2','NYC','NYE','NYZO','NZL','OBIC','OBSR','OBTC','OCB','OCE','OCEAN','OCN','OCP','OCT','OCTO','ODEM','ODEX','OG','OGN','OGO','OIKOS','OIN','OK','OKB','OKBBULL','OKT','OLT','OLY','OM','OMC','OMG','OMNI','ON','ONC','ONE','ONES','ONG','ONION','onLEXpa','ONS','ONT','ONTOLOGY-GAS','ONX-FINANCE','ONX','OOKI','OPAL','OPCT','OPEN-PLATFORM','OPEN','OPIUM','OPM','OPNN','OPUS','ORACLE-SYSTEM','ORACOLXOR','ORAI','ORB','ORBS','ORBYT','ORC','ORION','ORN','ORO','OSB','OSMO','OST','OTB','OTO','OUSD','OVR','OWC','OWN','OXT','PAC','PAI','PAID','PAK','PAN','PAR','PART','PARTY','PASC','PASS','PAT','PAX','PAXEX','PAXG','PAY','PAYT','PBR','PBTC','pBTC35A','PC','PCI','PCX','PEAK','PEARL','PEG','PENTA','PEPS','PERL','PERP','PERX','PEX','PFE','PFIZER-TOKENIZED-STOCK-FTX','PGN','PGO','PHA','PHB','PHILIPS-PAY-COIN','PHNX','PHONEUM','PHOSWAP','PHR','PHT','PI','PIB','PICA','PICKLE','PIE','PINK','PINT','PIPL','PIPT','PIRATE','PIVX','PIZZASWAP','PKB','PKT','PLANET','PLAT','PLAY','PLAYCHIP','PLAYDAPP','PLBT','PLC','PLEX','PLF','PLG','PLNC','PLOT','PLR','PLTC','PLU','PLURA','PLUS1','PMA','PMGT','PNK','PNT','PNX','PNY','POA','POC','POCKETNODE','POE','POL','POLA','POLC','POLIS','POLK','POLS','POLY','POND','PONZI','POOL','POOLZ','POP-NETWORK-TOKEN','PORTAL','POST','POSW','POT','POWER','POWR','POX','PPAY','PPBLZ','PPC','PPP','PPT','PRDX','PRDZ','PRE','PREMIA','PRIX','PRO','PROB','PROFILE-UTILITY-TOKEN','PROM','PROPS','PROS','PROTO','PROUD','PROXI','PRQ','PRT','PRVS','PSG','PSI','PSIX','PST','PTA','PTF','PTM','PUML','PURE','PUT','PVT','PWRB','PXC','PXI','PXL','PYN','PYPL','PYR','PYRK','PYRO','PZM','QARK','QASH','QBC','QBT','QBZ','QC','QCH','QCX','QDAO','QHC','QI','QIAN-KUN','QKC','QLC','QNT','QQQ','QRDO','QRK','QRL','QRX','QSP','QTCON','QTF','QTUM','QUAN','QUICK','QUIN','QWC','R2R','R34P','RAE','RAK','RAKU','RAMEN','RAMP','RANK','RAP','RARE','RARE1','RARI','RATECOIN','RATING','RAVEN','RAY','RAZOR','RBBT','RBC','RBIES','RBLX','RBN','RBT','RBTC','RBY','RC20','RCHAIN','RCN','RDD','RDN','REAP','RED','REEF','REL','RELEVANT','REM','REN','RENBTC','RENDOGE','RENFIL','RENTBERRY','REP','REQ','REV','REVV','REW','RFI','RFOX','RFR','RFUEL','RGT','RICK','RIF','RIGEL','RING','RING1','RINGX','RIO','RISE','RIT20','RITO','RKN','RLC','RLE','RLY','RM','RMPL','RNDR','RNO','RNT','ROAD','ROCK2','ROCKI','ROOBEE','ROOK','ROOM','ROOT','ROSE','ROUTE','ROX','ROYA','RPD','RPL','RPT','RPZX','RSR','RSV','RTE','RTH','RUC','RUFF','RUNE','RUP','RVC','RVN','RVT','RVX','RWN','RYO','S4F','SAFE','SAFECOIN','SAFEMOON','SAITAMA','SALE','SALT','SAN','SAND','SANDG','SANTOS','SAPP','SASHIMI','SATT','SATX','SAV3','SAVE','SAVG','SBD','SBDO','SBE','SBTC','SC','SCAP','SCC','SCH','SCL','SCP','SCR','SCRIV','SCRT','SCS','SCSX','SCV','SDT','SECO','SEELE','SEEN','SEFA','SELFKEY','SEMI','SENC','SEND','SENSO','SENT','SERO','SET','SETH','SEX','SFCP','SFD','SFG','SFI','SFM','SFP','SFT','SFUEL','SFX','SG','SGC','SGT','SHA','SHAKE','SHARD','SHARDUS','SHARE','SHB','SHD2','SHDW','SHE','SHENG','SHIB','SHIELD','SHIP','SHMN','SHND','SHR','SHROOM','SHVR','SIACASHCOIN','SIB','SIG','SIGN','SIN','SINS','SIX','SKB','SKC','SKEY','SKI','SKIN','SKL','SKLAY','SKM','SKY','SLM','SLNV2','SLP','SLS','SLV','SMART','SMARTCREDIT','SMBSWAP','SMLY','SMT','SNB','SNC','SND','SNET','SNGLSDAO','SNL','SNM','SNN','SNOW','SNRG','SNT','SNTVT','SNX','SOC','SOCC','SOCKS','SODA-COIN','SOFTBTC','SOL','SOLAPE','SOLID1','SOLO','SOLVE','SONG','SONO','SOP','SORA','SORA-VALIDATOR-TOKEN','SOS','SOUL','SPA','SPARTA','SPAZ','SPC','SPD','SPDR','SPELL','SPHTX','SPI','SPIRIT','SPIZ','SPK','SPORE','SPORTS','SPR','SPT','SPY','SQ','SQUID','SRIRACHA','SRK','SRM','SRN','SS','SSP','SSV','SSX','STA','STABLE-ASSET','STACY','STAK','STAKE','STAKECUBECOIN','STAR','START','STAX','STBU','STBZ','STC','STEEM','STETH','STIPEND','STM','STMX','STN','STO','STONK','STOP','STORJ','STOX','STP','STPL','STPT','STR','STRAX','STRONG','STS','STX','SUB','SUKU','SUMO','SUP','SUPER-BITCOIN','SUPERCOIN','SUPERFARM','SURE','SUSD','SUSHI','SUSHIBULL','SUTER','SWACE','SWAG','SWAP','SWC','SWFL','SWFTC','SWG','SWIFT','SWING','SWINGBY','SWISS','SWM','SWRV','SWSH','SWT','SWTH','SX','SXP','SXUT','SYLO','SYNC','SYS','SZC','TAD','TAG','TAI','TAJ','TAP','TAU','TBT','TBTC','TBX','TCASH','TCC','TCH','TCO','TCORE','TCP','TCT','TDP','TEAT','TEL','TELOS','TEM','TEMCO','TEN','TEND','TENET','TENT','TERA','TERC','TERN','TFL','TFT','TFUEL','TGAME','TH','THC','THE-BANK-COIN','THE','THEMIS-ORACLE','THEMIS','THETA','THORCHAIN-ERC20','THR','THX','TIC','TIGER','TIME','TIPS','TITAN','TIX','TKN','TKO','TKP','TKX','TKY','TLB','TLM','TLOS','TLW','TM2','TMC','TMED','TMN','TMTG','TN','TNB','TNC','TNS','TOC','TOK','TOKO','TOL','TOM','TOMO','TOMOBULL','TON-CRYSTAL','TONCOIN','TONE','TONS','TONTOKEN','TOP','TOPC','TOR','TORN','TOS','TOTO','TPT','TRA','TRAC','TRADE','TRAT','TRB','TRBT','TRC','TRCL','TREX','TRIBE','TRINITY-NETWORK-CREDIT','TRIO','TRISM','TRIX','TRND','TROLL','TRONBETDICE','TRONX','TROP','TROY1','TRP','TRST','TRTL','TRTT','TRU','TRUE','TRUST','TRV','TRX','TRXBEAR','TRXBULL','TRXC','TRXDOWN','TRXUP','TRYB','TSHP','TSL','TSLA','TSM','TST','TT','TTN','TTT','TUBE','TUNE','TUP','TUSD','TVK','TVNT','TWI','TWT','TWTR','TX','TXL','TZC','UAT','UBER','UBEX','UBIN','UBN','UBQ','UBT','UBTC','UBU','UBX','UBXT','UC','UCA','UCAP','UCM','UCN','UCO','UCR','UFO','UFR','UFT','UGAS','UHP','UIP','ULG','ULT','UMA','UMB','UMX','UNCL','UNCX','UNDG','UNFI','UNI','UNI1','UNIDX','UNIFI-PROTOCOL','UNIFI','UNIFY','UNISTAKE','UNIT-PROTOCOL-DUCK','UNIT','UNIUSD','UNKNOWN-FAIR-OBJECT','UNL','UNN','UNO','UNW','UOP','UOS','UPI','UPP','UPX','USDC','USDF','USDJ','USDK','USDN','USDQ','USDS','USDT','USDU','USDX','UST','UTED','UTK','UTT','UTU','UUU','UVU','UWL','VAI','VAIP','VAL','VALOR','VALUE','vBCH','VBIT','vBUSD','vDAI','VDG','VDL','vDOT','VEC2','VEE','VEIL','VELO','VEO','VERI','VEST','VET','VEX','vFIL','VGW','VGX','VIA','VIB','VIBE','VID','VIDT','VIDY','VIDYA','VIDYX','VIDZ','VIG','VIKING','VIKKY','VIN','VIPS','VITE','VIVID','VKNF','vLINK','VLS','VLT','vLTC','VLX','VN','VNDC','VNLA','VNT','VNXLU','VOLLAR','VOLT','VOLTZ','VOTE','VOX','VOYTEK-BEAR-COIN','VRA','VRO','VRSC','VRX','VSN','VSP','vSXP','VSYS','VTC','VTHO','vUSDC','vUSDT','VVS','VX','vXRP','VXV','vXVS','WAB','WABI','WAGE','WAN','WARP','WAVES','WAXE','WAXP','WBB','WBIND','WBNB','WBTC','WBX','WCC','WCK','WCO','WEB','WEBN','WELD','WELL','WEMIX','WEST','WETH','WFX','WG0','WGR','WGRT','WHALE','WHEN','WHITE','WHL','WHT','WICC','WIFI','WIKEN','WIN','WING','WINGS','WINGSHOP','WIS','WISE','WISH','WIX','WLEO','WNXM','WOM','WON','WOO','WORLD','WOW','WOWS','WOWSWAP','WOZX','WPP','WPR','WPX','WRC','WRLD','WRX','WSCRT','WTC','WTN','WXC','WXMR','WXT','X42','XAG','XAMP','XANK','XAP','XAS','XAU','XAUR','XAUT','XAUTBEAR','XAUTBULL','XBC','XBI','XBP','XBTC21','XBY','XCAD','XCASH','XCF','XCM','XCO','XCP','XCUR','XDAG','XDB','XDC','XDN','XDNA','XED','XEM','XEP','XEQ','XFI','XFT','XFUEL','XFUND','XGM','XGS','XHI','XHV','XIO','XIOT','XLA','XLAB','XLM','XLPG','XLR','XLT','XMC','XMON','XMR','XMV','XMX','XMY','XNC','XNO','XNO1','XNODE','XNV','XOR','XPA','XPC','XPD','XPM','XPN','XPR','XPTOKEN-IO','XPTX','XPX','XPY','XQC','XQN','XRA','XRC','XRP','XRPBEAR','XRPBULL','XRPDOWN','XRPUP','XRT','XRU','XSGD','XSH','XSPC','XSR','XSTABLE-PROTOCOL','XTK','XTP','XTRM','XTZ','XTZBEAR','XTZBULL','XUC','XUEZ','XUSD','XVG','XVIX','XVS','XWC','XWP','XXA','XYO','YAM','YAMV1','YAS','YAX','yBAN','YBO','YCC','YCE','YEA','YEC','YEE','YeFiM','YF-DAI','YFA','YFB2','YFBETA','YFBT','YFBTC','YFD','YFDOT','YFE','YFFI','YFFII','YFFS','YFI','YFIA','YFIAG','YFICG','YFID','YFIE','YFII','YFIM','YFIX','YFL','YFMS','YFN','YFO','YFOX','YFPRO','YFRM','YGG','YI12','YIELD','YLC','YLD','YLFI','YO','YOP','YOT','YOU','YOUC','YOYOW','YSEC','YSR','YT','YTN','yTSLA','YUSRA','YVS','YYFI','ZANO','ZAP','ZASH','ZB','ZCL','ZCN','ZCR','ZDEX','ZDR','ZDX','ZEBI','ZEC','ZEE','ZEFU','ZEL','ZEN','ZENI','ZEON','ZER','ZERO','ZET','ZEUS','ZFL','ZIK','ZIL','ZIPT','ZKS','ZLP','ZLW','ZMT','ZNN','ZNT','ZNY','ZNZ','ZOC','ZOOM-PROTOCOL','ZOOM','ZORA','ZPAE','ZPR','ZPT','ZRX','ZSC','ZT','ZUC','ZUM','ZUMCOIN','ZUR','ZUSD','ZUT','ZXC','ZYD','ZYN','ZYRO'];
        }

        /**
         * Get array of currency codes
         * @return array
         */
        public function get_forex_codes()
        {
            return ['ANG','ARS','AUD','BRL','CAD','CHF','CLP','CNY','CZK','DKK','EUR','GBP','HKD','HUF','IDR','ILS','INR','ISK','JPY','KRW','LKR','MXN','MXV','MYR','NGN','NOK','NZD','PEN','PHP','PKR','PLN','RON','RUB','SAR','SEK','SGD','THB','TOP','TRY','TWD','USD','UYU','VND','XDR','ZAC','ZAR'];
        }
    }
}


if(class_exists('EOD_API')) {
    function eod_api()
    {
        global $eod_api;

        // Instantiate only once.
        if ( ! isset( $eod_api ) ) {
            $eod_api = new EOD_API();
        }
        return $eod_api;
    }

    // Instantiate.
    eod_api();
}