<?php

defined( 'ABSPATH' ) || die();

/**
 * Get a specific View WPPedia checks for view files in the following order:
 * - path/to/childtheme/wppedia/template-name.php
 * - path/to/theme/wppedia/template-name.php
 * - path/to/wppedia/templates/template-name.php
 * 
 * @since 1.0.0
 */
function wppedia_locate_template(string $template_name, string $template_path = '', string $default_path = '') {
	
	if ( ! $template_path ) { 
		$template_path = WPPedia()->template_path();
	}
	
	if ( ! $default_path ) { 
		$default_path = WPPedia()->plugin_path() . 'templates/'; 
	} 
	
	// Check the current Theme and Childtheme for WPPedia Templates
	$template = locate_template(trailingslashit( $template_path ) . $template_name);
	
	// If no template was found fall back to the default Templates
	if ( ! $template || WPPedia_TEMPLATE_DEBUG_MODE ) { 
		$template = $default_path . $template_name; 
	}
	
	return apply_filters( 'wppedia_locate_template', $template, $template_name, $template_path );
	
}

/** 
 * Get template part. 
 * 
 * WPPedia_TEMPLATE_DEBUG_MODE will prevent overrides in themes from taking priority. 
 * 
 * @access public 
 * @param mixed $slug 
 * @param string $name (default: '') 
 */ 
function wppedia_get_template_part( $slug, $name = '' ) { 

  $template = ''; 

  // Look in yourtheme/slug-name.php and yourtheme/wppedia/slug-name.php 
  if ( $name && ! WPPedia_TEMPLATE_DEBUG_MODE ) { 
    $template = locate_template( WPPedia()->template_path() . "{$slug}-{$name}.php" ); 
  } 

  // Get default slug-name.php 
  if ( ! $template && $name && file_exists( WPPedia()->plugin_path() . "templates/{$slug}-{$name}.php" ) ) { 
    $template = WPPedia()->plugin_path() . "templates/{$slug}-{$name}.php"; 
  }

  // If template file doesn't exist, look in and yourtheme/wppedia/slug.php 
  if ( ! $template && ! WPPedia_TEMPLATE_DEBUG_MODE ) { 
    $template = locate_template( WPPedia()->template_path() . "{$slug}.php" ); 
  } 

	// Get default slug.php 
	if ( ! $template && ! $name && file_exists( WPPedia()->plugin_path() . "templates/{$slug}.php" ) ) {
		$template = WPPedia()->plugin_path() . "templates/{$slug}.php"; 
	}

  // Allow 3rd party plugins to filter template file from their plugin. 
  $template = apply_filters( 'wppedia_get_template_part', $template, $slug, $name ); 

  if ( $template ) { 
    load_template( $template, false ); 
  }

}

/**
 * Get the current page title
 *
 * @param  bool $echo Should echo title.
 * 
 * @since 1.2.1
 */
function wppedia_page_title( $echo = true ) {
	global $post;

	if ( is_search() ) { 
		$page_title = sprintf( __( 'Search results: “%s”', 'wppedia' ), get_search_query() );
		if ( get_query_var( 'paged' ) )
			$page_title .= sprintf( __( ' – Page %s', 'wppedia' ), get_query_var( 'paged' ) );
	} else if ( is_tax() ) { 
		$page_title = single_term_title( "", false );
	} else if ( is_wppedia_frontpage() ) {
		$front_page_id = wppedia_get_page_id( 'front' );
		$page_title = get_the_title( $front_page_id );
	} else if ( is_wppedia_archive() ) {
		$page_title = __('Glossary', 'wppedia');
	} else {
		$page_title = get_the_title();
	}

	$page_title = apply_filters( 'wppedia_page_title', $page_title ); 

	if ( $echo ) 
		echo $page_title; 
	else 
		return $page_title;
}

/**
 * Get the excerpt with fallback generated from the content
 * 
 * @param WP_Post|int $post - Post ID or object
 * @param bool $force_balanced_tags - If true try to keep bold or other formatting
 * 
 * @since 1.0.0
 */
function get_the_excerpt_wppedia( $post = null, int $excerpt_length = 40, bool $force_balanced_tags = false ) {
	
	$str = '';
	
	// setup Postdata
  $post = get_post( $post );
  if ( empty( $post ) )
    return;
	
	if ( ! has_excerpt( $post ) ) {
		
		// Get the Post Content (formatted)
		setup_postdata( $post );
		$str = get_the_content( null, false, $post );
		wp_reset_postdata( $post );
		
		// Check if Text is not empty
		if ( '' != $str && $str ) {
			
			// Remove Shortcodes
			$str = \strip_shortcodes( $str );
			$str = str_replace(']]&gt;', ']]&gt;', $str);

			// Remove unwanted tags
			$str = strip_tags( 
				$str, 
				[ 
					'em', 
					'i', 
					'strong', 
					'b',
					'p',
					'br'
				] 
			);

			// Trim and format the string
			if ( $force_balanced_tags )
				$str = force_balance_tags( html_entity_decode( wp_trim_words( htmlentities( $str ), $excerpt_length, null ) ) );
			else
				$str = wp_trim_words( $str, $excerpt_length, null );
			
		}
		
	} else {
		
		// If an excerpt was specified just add some p tags
		$str = wpautop( $post->post_excerpt );
		
	}
	
	return apply_filters( 'wppedia_tooltip_excerpt', $str );
	
}

/**
 * Display the autogenerated excerpt
 * 
 * @param WP_Post|int $post - Post ID or object
 * 
 * @since 1.0.0
 */
function the_excerpt_wppedia( $post = null, int $excerpt_length = 40, bool $force_balanced_tags = false) {
	echo get_the_excerpt_wppedia( $post, $excerpt_length, $force_balanced_tags );
}

/**
 * Template functions related to the searchform
 * 
 * @since 1.0.0
 */

/**
 * Get searchform attributes
 * 
 * @param array $attrs - Additional Attributes
 * @param boolean $tostring - Whether to render the Attributes as a string or return an Array
 * 
 * @since 1.0.0
 */
function wppedia_get_search_form_attrs( array $attrs = [], bool $tostring = true ) {
	
	$post_type = wppedia_get_post_type();
	$searchUrl = get_post_type_archive_link( $post_type );
	
	/**
	 * Predefined attributes
	 */
	$_attrs = [
		'role'		=> apply_filters( 'wppedia_searchform_attrs__role', 'search' ),
		'method' 	=> apply_filters( 'wppedia_searchform_attrs__method', 'GET' ),
		'class' 	=> apply_filters( 'wppedia_searchform_attrs__class', 'search-form wppedia-search' ),
		'id' 			=> apply_filters( 'wppedia_searchform_attrs__id', 'wppedia_searchform' ),
		'action' 	=> $searchUrl
	];
	
	$attrs = array_merge( $attrs, $_attrs );
	
	if ( $tostring ) {
		
		$final = '';
		
		$attr_index = 0;
		$attr_count = count( $attrs );
		foreach ( $attrs as $k => $v ) {
			$attr_index++;
			$final .= $k . '="' . $v . '"';
			if ( $attr_index < $attr_count )
				$final .= ' ';
		}
		
		return $final;
		
	}
	
	return $attrs;
	
}
